<?php

namespace WM\Search;

use WM\Search\Response\Response;
use WM\Search\Response\Suggest;
use Elasticsearch\Client as ElasticsearchClient;
use Elasticsearch\ClientBuilder;
use Psr\Log\LoggerInterface;

class Client
{
    /**
     * Final score is _score * (status^STATUS_WEIGHT)
     * _score is ES score
     * status is a computed status from internal datas
     */
    public const float STATUS_WEIGHT = 1.5;

    public const array DEFAULT_ENTITY_TYPES = ['movie', 'series', 'theater', 'person'];

    private readonly ?array $hosts;

    private ?LoggerInterface $logger = null;

    private ?ElasticsearchClient $elasticsearch = null;

    /**
     * @param array|null $hosts ['host1:port1', 'host2:port2']
     */
    public function __construct(private readonly string $apiKeyPair, ?array $hosts = null)
    {
        if (null === $hosts) {
            $hosts = ['127.0.0.1:9200'];
        }

        $this->hosts = $hosts;
    }

    public function setLogger(LoggerInterface $logger): void
    {
        $this->logger = $logger;
    }

    public function getElasticsearch(): ElasticsearchClient
    {
        if (!$this->elasticsearch instanceof ElasticsearchClient) {
            $clientBuilder = ClientBuilder::create()
                ->setHosts($this->hosts);

            if ($this->logger instanceof LoggerInterface) {
                $clientBuilder->setLogger($this->logger);
            }

            if ($this->apiKeyPair !== '' && $this->apiKeyPair !== '0') {
                $apiKeyPairDecoded = base64_decode($this->apiKeyPair);
                if (false === $apiKeyPairDecoded) {
                    throw new \InvalidArgumentException('Invalid API key pair');
                }

                $apiKeyPair = explode(':', $apiKeyPairDecoded);
                if (2 !== count($apiKeyPair)) {
                    throw new \InvalidArgumentException('Invalid API key pair');
                }

                $clientBuilder->setApiKey($apiKeyPair[0], $apiKeyPair[1]);
            }

            $this->elasticsearch = $clientBuilder->build();
        }

        return $this->elasticsearch;
    }

    public function autocomplete(string $text, string $brand, array $entityTypes, array $excludeEntities = []): Response
    {
        $request = new Request();
        $request->setQuery($text);
        $request->setBrand($brand);
        $request->setExcludedEntities($excludeEntities);
        $request->setSize(10);
        $request->setIsAutocomplete(true);
        $request->setIsBrowsableOnly(true);
        $request->setEntityTypes($entityTypes === [] ? self::DEFAULT_ENTITY_TYPES : $entityTypes);

        return $this->search($request);
    }

    public function search(Request $request): Response
    {
        $startTime = microtime(true);

        // direct results
        $response = $this->legacySearch($request);
        if ($response->count() > 0) {
            $totalTime = round((microtime(true) - $startTime) * 1000);
            $response->setTotalTime(($totalTime));

            return $response;
        }

        if ($request->allowSpellChecking()) {
            // phrase suggestion
            $suggest = $this->phraseSuggestion($request->getQuery(), $request->getBrand(), 1, false);
            if ($suggest->count() > 0) {

                $proposalRequest = clone $request;
                $proposalRequest->setQuery($suggest->getFirstProposalText());

                $response = $this->legacySearch($proposalRequest);
                $response->setSuggested(true);
                $response->setSuggestion($suggest->getFirstProposalText());

                $totalTime = round((microtime(true) - $startTime) * 1000);
                $response->setTotalTime(($totalTime));

                return $response;
            }

            // terms suggestion
            $terms = preg_split('#[^a-zA-Z0-9]+#', $request->getQuery());
            $terms = array_filter($terms, fn($item) => strlen((string) $item) >= 2);

            $allTerms = [];
            $termsSuggestion = $this->termsSuggestion($terms, $request->getBrand(), $request->getSize());
            foreach ($termsSuggestion->getProposals() as $key => $proposal) {
                $allTerms[$key] = $proposal->getText();
            }
            $termsSuggestion->addTermWithNoSuggestion($terms, $allTerms);

            $response = $this->permissiveSearch(implode(' ', $allTerms), count($terms), $request->getBrand(), $request->getEntityTypes());

            if ($response->count() > 0) {
                $totalTime = round((microtime(true) - $startTime) * 1000);
                $response->setSuggested(true);
                $response->setTotalTime(($totalTime));

                return $response;
            }
        }

        // no results
        $response = new Response([]);
        $totalTime = round((microtime(true) - $startTime) * 1000);
        $response->setTotalTime(($totalTime));

        return $response;
    }

    public function legacySearch(Request $request): Response
    {
        $disMaxQueries = [
            [
                'match' => [
                    'search_label.keyword' => [
                        'query' => $request->getQuery(),
                        'boost' => 40,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'search_label.lowercase' => [
                        'query' => $request->getQuery(),
                        'boost' => 20,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'label.keyword' => [
                        'query' => $request->getQuery(),
                        'boost' => 40,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'original_label.keyword' => [
                        'query' => $request->getQuery(),
                        'boost' => 40,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'label.lowercase' => [
                        'query' => $request->getQuery(),
                        'boost' => 20,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'original_label.lowercase' => [
                        'query' => $request->getQuery(),
                        'boost' => 20,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'text_search_data.keyword' => [
                        'query' => $request->getQuery(),
                        'boost' => 5,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'text_search_data.lowercase' => [
                        'query' => $request->getQuery(),
                        'boost' => 2,
                        'operator' => 'and'
                    ]
                ]
            ]
        ];

        if ($request->isAutocomplete()) {
            $disMaxQueries[] = [
                'match' => [
                    'search_label.autocomplete' => [
                        'query' => $request->getQuery(),
                        'boost' => 10,
                        'operator' => 'and'
                    ]
                ]
            ];
            $disMaxQueries[] = [
                'match' => [
                    'label.autocomplete_full' => [
                        'query' => $request->getQuery(),
                        'boost' => 15,
                        'operator' => 'and'
                    ]
                ]
            ];
            $disMaxQueries[] = [
                'match' => [
                    'original_label.autocomplete_full' => [
                        'query' => $request->getQuery(),
                        'boost' => 15,
                        'operator' => 'and'
                    ]
                ]
            ];
            $disMaxQueries[] = [
                'match' => [
                    'text_search_data.autocomplete' => [
                        'query' => $request->getQuery(),
                        'boost' => 1,
                        'operator' => 'and'
                    ]
                ]
            ];
        }

        $musts = [
            [
                'dis_max' => [
                    'queries' => $disMaxQueries,
                    'tie_breaker' => 1.0
                ]
            ]
        ];

        $filters = [];
        if ($request->hasEntityTypes()) {
            $filters[] = [
                'terms' => [
                    'entity_type' => $request->getEntityTypes()
                ]
            ];
        }

        // starts/ends at (for news)
        $filters[] = [
            'bool' => [
                'should' => [
                    [
                        'range' => [
                            'starts_at' => [
                                'lte' => (new \DateTime())->format('c')
                            ]
                        ]
                    ], [
                        'bool' => [
                            'must_not' => [
                                [
                                    'exists' => [
                                        'field' => 'starts_at'
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];
        $filters[] = [
            'bool' => [
                'should' => [
                    [
                        'range' => [
                            'ends_at' => [
                                'gte' => (new \DateTime())->format('c')
                            ]
                        ]
                    ], [
                        'bool' => [
                            'must_not' => [
                                [
                                    'exists' => [
                                        'field' => 'ends_at'
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $mustNot = [];
        if ($request->hasExcludedEntities()) {
            $mustNot[] = [
                'terms' => [
                    '_id' => $request->getExcludedEntities()
                ]
            ];
        }
        if ($request->hasExcludedGenres()) {
            $mustNot[] = [
                'terms' => [
                    'genres' => $request->getExcludedGenres(),
                ],
            ];
        }
        if ($request->isBrowsableOnly()) {
            $filters[] = [
                'term' => [
                    'browsable' => true
                ]
            ];
        }


        $query = [
            'from' => $request->getFrom(),
            'size' => $request->getSize(),
            'query' => [
                'function_score' => [
                    'boost_mode' => 'replace',
                    'script_score' => [
                        'script' => [
                            'id' => 'sort-legacy-search',
                            'params' => [
                                'weight' => self::STATUS_WEIGHT
                            ]
                        ]
                    ],
                    'query' => [
                        'bool' => [
                            'must' => $musts,
                            'filter' => $filters,
                            'must_not' => $mustNot,
                        ]
                    ]
                ]
            ],
            'sort' => $request->getSort(),
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($request->getBrand())),
                'type' => null,
                'body' => $query
            ]);

        return new Response($results);
    }

    /**
     * Return results which contains at least $minmumShouldMatch words of $terms
     *
     *
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-minimum-should-match.html
     */
    public function permissiveSearch(
        string $terms,
        string $minimumShouldMatch,
        string $brand,
        array $entities = []
    ): Response {
        $filters = [];
        if ($entities !== []) {
            $filters[] = [
                'terms' => [
                    'entity_type' => $entities
                ]
            ];
        }

        $query = [
            'query' => [
                'bool' => [
                    'must' => [
                        [
                            'match' => [
                                'search_label.lowercase' => [
                                    'query' => $terms,
                                    'operator' => 'or',
                                    'minimum_should_match' => $minimumShouldMatch
                                ]
                            ]
                        ]
                    ],
                    'filter' => $filters
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => null,
                'body' => $query
            ]);
        return new Response($results);
    }

    /**
     *
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/search-suggesters-phrase.html
     */
    public function phraseSuggestion(
        string $text,
        string $brand,
        int $size = 10,
        bool $prune = true,
        ?array $highlight = null
    ): Suggest {
        $highlightQuery = [];
        if (null !== $highlight && is_array($highlight)) {
            $highlightQuery = [
                'highlight' => [
                    'pre_tag' => $highlight[0],
                    'post_tag' => $highlight[1]
                ]
            ];
        }

        $query = [
            'suggest' => [
                'text' => $text,
                'proposals' => [
                    'phrase' => array_merge([
                        'field' => 'search_label.trigram',
                        'size' => $size,
                        'max_errors' => 4,
                        'direct_generator' => [
                            [
                                'field' => 'search_label.trigram',
                                'suggest_mode' => 'always'
                            ], [
                                'field' => 'search_label.reverse',
                                'suggest_mode' => 'always',
                                'pre_filter' => 'reverse_analyzer',
                                'post_filter' => 'reverse_analyzer'
                            ]
                        ],
                        'collate' => [
                            'query' => [
                                'source' => [
                                    'match' => [
                                        '{{field_name}}' => [
                                            'query' => '{{suggestion}}',
                                            'analyzer' => 'lowercase_analyzer',
                                            'operator' => 'and'
                                        ]
                                    ]
                                ]
                            ],
                            'params' => [
                                'field_name' => 'search_label.lowercase'
                            ],
                            'prune' => $prune
                        ]
                    ], $highlightQuery)
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => null,
                'body' => $query
            ]);
        return new Suggest($results);
    }

    /**
     * @deprecated
     * @see self::termsSuggestion()
     *
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/search-suggesters-term.html
     */
    public function termSuggestion(string $term, string $brand, int $size = 10): Suggest
    {
        $query = [
            'suggest' => [
                'proposals' => [
                    'text' => $term,
                    'term' => [
                        'field' => 'search_label.lowercase',
                        'analyzer' => 'lowercase_analyzer',
                        'sort' => 'score',
                        'suggest_mode' => 'missing',
                        'max_edits' => 2
                    ]
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => null,
                'body' => $query
            ]);

        return new Suggest($results, false);
    }

    /**
     * @param string $term
     *
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/search-suggesters.html#term-suggester
     */
    public function termsSuggestion(array $terms, string $brand, int $size = 10): Suggest
    {
        $suggesters = [];
        foreach ($terms as $key => $term) {
            $suggesters["proposals-{$key}"] = [
                'text' => $term,
                'term' => [
                    'field' => 'search_label.lowercase',
                    'analyzer' => 'lowercase_analyzer',
                    'sort' => 'score',
                    'suggest_mode' => 'missing',
                    'max_edits' => 2,
                    'size' => $size,
                ]
            ];
        }

        $query = [
            'index' => sprintf('search_%s', strtolower($brand)),
            'type' => null,
            'body' => [
                'suggest' => $suggesters
            ]
        ];

        $results = $this->getElasticsearch()
            ->search($query);

        return new Suggest($results);
    }

    /**
     * @deprecated
     */
    public function findById(string $brand, array $entities = []): Response
    {
        $filters = [];
        if ($entities !== []) {
            $filters[] = [
                'terms' => [
                    '_id' => $entities
                ]
            ];
        }

        $query = [
            'query' => [
                'bool' => [
                    'filter' => $filters,
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => null,
                'body' => $query
            ]);

        return new Response($results);
    }

    public function findByInternalId(string $brand, int $entityId, string $entityType): Response
    {
        $query = [
            'query' => [
                'bool' => [
                    'filter' => [
                        [
                            'term' => [
                                'entity_id' => $entityId
                            ]
                        ], [
                            'term' => [
                                'entity_type' => $entityType
                            ]
                        ]
                    ],
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => null,
                'body' => $query
            ]);

        return new Response($results);
    }
}
