<?php

namespace WM\Search;

use WM\Search\Exception\BadRequestException;
use WM\Search\Response\Proposal;
use WM\Search\Response\Response;
use WM\Search\Response\Suggest;
use Elasticsearch\ClientBuilder;
use Psr\Log\LoggerInterface;

class Client
{
    /**
     * Final score is _score * (status^STATUS_WEIGHT)
     * _score is ES score
     * status is a computed status from internal datas
     */
    const STATUS_WEIGHT = 1.5;

    /**
     * @var array
     */
    private $hosts;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var \Elasticsearch\Client
     */
    private $elasticsearch;

    /**
     * @param array|null $hosts ['host1:port1', 'host2:port2']
     */
    public function __construct(array $hosts = null)
    {
        if (null === $hosts) {
            $hosts = ['127.0.0.1:9200'];
        }

        $this->hosts = $hosts;
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @return \Elasticsearch\Client
     */
    public function getElasticsearch()
    {
        if (null === $this->elasticsearch) {
            $clientBuilder = ClientBuilder::create()
                ->setHosts($this->hosts);

            if (null !== $this->logger) {
                $clientBuilder->setLogger($this->logger);
            }


            $this->elasticsearch = $clientBuilder->build();
        }

        return $this->elasticsearch;
    }

    /**
     * @param string $text
     * @param string $brand
     * @param array $excludeEntities
     *
     * @return Response
     * @throws BadRequestException
     */
    public function autocomplete(string $text, string $brand, array $excludeEntities = []): Response
    {
        return $this->search($text, $brand, [], $excludeEntities, [], [],0, 10, true);
    }

    /**
     * @param string $text
     * @param string $brand
     * @param array  $entities
     * @param array  $excludeEntities
     * @param array  $excludedGenres
     * @param array  $sort
     * @param int    $size
     * @param int    $from
     * @param bool   $autocomplete
     *
     * @return Response
     * @throws BadRequestException
     */
    public function search(
        string $text,
        string $brand,
        array $entities = [],
        array $excludeEntities = [],
        array $excludedGenres = [],
        array $sort = [],
        int $from = 0,
        int $size = 10,
        bool $autocomplete = false
    ): Response {
        $startTime = microtime(true);

        // direct results
        $response = $this->legacySearch($text, $brand, $entities, $excludeEntities, $excludedGenres, $sort, $from, $size, $autocomplete);
        if ($response->count() > 0) {
            $totalTime = round((microtime(true) - $startTime)*1000);
            $response->setTotalTime(($totalTime));

            return $response;
        }

        // phrase suggestion
        $suggest = $this->phraseSuggestion($text, $brand,  1, false);
        if ($suggest->count() > 0) {
            $response = $this->legacySearch(
                $suggest->getFirstProposalText(),
                $brand,
                $entities,
                $excludeEntities,
                $excludedGenres,
                $sort,
                $from,
                $size,
                $autocomplete
            );
            $response->setSuggested(true);
            $response->setSuggestion($suggest->getFirstProposalText());

            $totalTime = round((microtime(true) - $startTime)*1000);
            $response->setTotalTime(($totalTime));

            return $response;
        }

        // terms suggestion
        $terms = preg_split('#[^a-zA-Z0-9]+#', $text);
        $terms = array_filter($terms, function($item) {
            if (strlen($item) < 2) {
                return false;
            }

            return true;
        });

        $allTerms = [];
        foreach ($terms as $term) {
            $termSuggestion = $this->termSuggestion($term, $brand, $size);

            if (!$termSuggestion->hasProposal()) {
                $allTerms[] = $term;
                continue;
            }

            /** @var Proposal $proposal */
            foreach ($termSuggestion->getProposals() as $proposal) {
                $allTerms[] = $proposal->getText();
            }
        }

        $response = $this->permissiveSearch(implode(' ', $allTerms), count($terms), $brand, $entities);

        if ($response->count() > 0) {
            $totalTime = round((microtime(true) - $startTime)*1000);
            $response->setSuggested(true);
            $response->setTotalTime(($totalTime));

            return $response;
        }

        // no results
        $response = new Response([]);
        $totalTime = round((microtime(true) - $startTime)*1000);
        $response->setTotalTime(($totalTime));

        return $response;
    }

    /**
     * @param string $text
     * @param string $brand
     * @param array  $entities
     * @param array  $excludeEntities
     * @param array  $excludedGenres
     * @param array  $sort
     * @param int    $size
     * @param int    $from
     * @param bool   $autocomplete
     *
     * @return Response
     */
    public function legacySearch(
        string $text,
        string $brand,
        array $entities = [],
        array $excludeEntities = [],
        array $excludedGenres = [],
        array $sort = [],
        int $from = 0,
        int $size = 10,
        bool $autocomplete = false
    ): Response {
        $disMaxQueries = [
            [
                'match' => [
                    'search_label.keyword' => [
                        'query' => $text,
                        'boost' => 20,
                        'operator' => 'and'
                    ]
                ]
            ],
            [
                'match' => [
                    'search_label.lowercase' => [
                        'query' => $text,
                        'boost' => 10,
                        'operator' => 'and'
                    ]
                ]
            ]
        ];

        if ($autocomplete) {
            $disMaxQueries[] = [
                'match' => [
                    'search_label.autocomplete' => [
                        'query' => $text,
                        'boost' => 2,
                        'operator' => 'and'
                    ]
                ]
            ];
        }

        $musts = [
            [
                'dis_max' => [
                    'queries' => $disMaxQueries
                ]
            ]
        ];

        $filters = [];
        if (!empty($entities)) {
            $filters[] = [
                'terms' => [
                    'entity_type' => $entities
                ]
            ];
        }

        $mustNot = [];
        if (!empty($excludeEntities)) {
            $mustNot[] = [
                'terms' => [
                    '_id' => $excludeEntities
                ]
            ];
        }
        if (!empty($excludedGenres)) {
            $mustNot[] = [
                'terms' => [
                    'genres' => $excludedGenres,
                ],
            ];
        }



        $query = [
            'from' => $from,
            'size' => $size,
            'query' => [
                'function_score' => [
                    'boost_mode' => 'replace',
                    'script_score' => [
                        'script' => [
                            'id' => 'sort-legacy-search',
                            'params' => [
                                'weight' => self::STATUS_WEIGHT
                            ]
                        ]
                    ],
                    'query' => [
                        'bool' => [
                            'must' => $musts,
                            'filter' => $filters,
                            'must_not' => $mustNot,
                        ]
                    ]
                ]
            ],
            'sort' => $sort,
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => 'search',
                'body' => $query
            ]);

        return new Response($results);
    }

    /**
     * Return results which contains at least $minmumShouldMatch words of $terms
     *
     * @param string $terms
     * @param string $minimumShouldMatch
     * @param string $brand
     * @param array $entities
     *
     * @return Response
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-minimum-should-match.html
     */
    public function permissiveSearch(
        string $terms,
        string $minimumShouldMatch,
        string $brand,
        array $entities = []
    ): Response {
        $filters = [];
        if (!empty($entities)) {
            $filters[] = [
                'terms' => [
                    'entity_type' => $entities
                ]
            ];
        }

        $query = [
            'query' => [
                'bool' => [
                    'must' => [
                        [
                            'match' => [
                                'search_label.lowercase' => [
                                    'query' => $terms,
                                    'operator' => 'or',
                                    'minimum_should_match' => $minimumShouldMatch
                                ]
                            ]
                        ]
                    ],
                    'filter' => $filters
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => 'search',
                'body' => $query
            ]);

        return new Response($results);
    }

    /**
     * @param string $text
     * @param string $brand
     * @param int $size
     * @param bool $prune
     * @param array|null $highlight
     *
     * @return Suggest
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/search-suggesters-phrase.html
     */
    public function phraseSuggestion(
        string $text,
        string $brand,
        int $size = 10,
        bool $prune = true,
        array $highlight = null
    ): Suggest {
        $highlightQuery = [];
        if (null !== $highlight && is_array($highlight)) {
            $highlightQuery = [
                'highlight' => [
                    'pre_tag' => $highlight[0],
                    'post_tag' => $highlight[1]
                ]
            ];
        }

        $query = [
            'suggest' => [
                'text' => $text,
                'proposals' => [
                    'phrase' => array_merge([
                        'field' => 'search_label.trigram',
                        'size' => $size,
                        'max_errors' => 4,
                        'direct_generator' => [
                            [
                                'field' => 'search_label.trigram',
                                'suggest_mode' => 'always'
                            ], [
                                'field' => 'search_label.reverse',
                                'suggest_mode' => 'always',
                                'pre_filter' => 'reverse_analyzer',
                                'post_filter' => 'reverse_analyzer'
                            ]
                        ],
                        'collate' => [
                            'query' => [
                                'source' => [
                                    'match' => [
                                        '{{field_name}}' => [
                                            'query' => '{{suggestion}}',
                                            'analyzer' => 'lowercase_analyzer',
                                            'operator' => 'and'
                                        ]
                                    ]
                                ]
                            ],
                            'params' => [
                                'field_name' => 'search_label.lowercase'
                            ],
                            'prune' => $prune
                        ]
                    ], $highlightQuery)
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => 'search',
                'body' => $query
            ]);

        return new Suggest($results);
    }

    /**
     * @param string $term
     * @param string $brand
     * @param int $size
     *
     * @return Suggest
     *
     * @link https://www.elastic.co/guide/en/elasticsearch/reference/current/search-suggesters-term.html
     */
    public function termSuggestion(string $term, string $brand, int $size = 10): Suggest
    {
        $query = [
            'suggest' => [
                'proposals' => [
                    'text' => $term,
                    'term' => [
                        'field' => 'search_label.lowercase',
                        'analyzer' => 'lowercase_analyzer',
                        'sort' => 'score',
                        'suggest_mode' => 'missing',
                        'max_edits' => 2
                    ]
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => 'search',
                'body' => $query
            ]);

        return new Suggest($results);
    }

    /**
     * @param string $brand
     * @param array $entities
     *
     * @return Response
     */
    public function findById(string $brand, array $entities = [])
    {
        $filters = [];
        if (!empty($entities)) {
            $filters[] = [
                'terms' => [
                    '_id' => $entities
                ]
            ];
        }

        $query = [
            'query' => [
                'bool' => [
                    'filter' => $filters,
                ]
            ]
        ];

        $results = $this->getElasticsearch()
            ->search([
                'index' => sprintf('search_%s', strtolower($brand)),
                'type' => 'search',
                'body' => $query
            ]);

        return new Response($results);
    }
}
