<?php

namespace WM\Search\Response;

use WM\Search\Exception\MethodNotAllowedException;

class Item implements \ArrayAccess, \JsonSerializable
{
    /**
     * @var string
     */
    private $_index;

    /**
     * @var string
     */
    private $_type;

    /**
     * @var string
     */
    private $_id;

    /**
     * @var float
     */
    private $_score;

    /**
     * @var array
     */
    private $_source;

    /**
     * @param array $hit
     */
    public function __construct(array $hit)
    {
        $this->_index = $hit['_index'];
        $this->_type = $hit['_type'];
        $this->_id = $hit['_id'];
        $this->_score = $hit['_score'];
        $this->_source = $hit['_source'];
    }

    /**
     * @return string
     */
    public function getIndex(): string
    {
        return $this->_index;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->_type;
    }

    /**
     * @return string
     */
    public function getId(): string
    {
        return $this->_id;
    }

    /**
     * https://www.elastic.co/guide/en/elasticsearch/guide/current/_sorting.html#_sorting_by_field_values
     * @return float | null
     */
    public function getScore(): ?float
    {
        return $this->_score;
    }

    /**
     * @return array
     */
    public function getSource(): array
    {
        return $this->_source;
    }

    /**
     * @param mixed $offset
     *
     * @return bool
     */
    public function offsetExists($offset)
    {
        return isset($this->_source[$offset]);
    }

    /**
     * @param mixed $offset
     *
     * @return mixed|null
     */
    public function offsetGet($offset)
    {
        return $this->_source[$offset] ?? null;
    }

    /**
     * @param mixed $offset
     * @param mixed $value
     *
     * @throws MethodNotAllowedException
     */
    public function offsetSet($offset, $value)
    {
        throw new MethodNotAllowedException('Cannot modify a response item field');
    }

    /**
     * @param mixed $offset
     *
     * @throws MethodNotAllowedException
     */
    public function offsetUnset($offset)
    {
        throw new MethodNotAllowedException('Cannot delete a response item field');
    }

    /**
     * @return bool|string
     */
    public function jsonSerialize()
    {
        return json_encode([
            '_index' => $this->_index,
            '_type' => $this->_type,
            '_id' => $this->_id,
            '_score' => $this->_score,
            '_source' => $this->_source
        ]);
    }
}