<?php

namespace WM\Library\Monolog\Processor;

use Monolog\Processor\WebProcessor as BaseWebProcessor;

/**
 * Adds a unique identifier issued from header X-Unique-Id set by HAProxy into records
 */
class WebProcessor extends BaseWebProcessor
{
    public const DEFAULT_UNIQUE_ID_LENGTH = 10;

    /**
     * Default fields
     *
     * Array is structured as [key in record.extra => key in $serverData]
     *
     * @var array
     */
    protected $extraFields = array(
        'url'         => 'REQUEST_URI',
        'ip'          => 'REMOTE_ADDR',
        'http_method' => 'REQUEST_METHOD',
        'server'      => 'SERVER_NAME',
        'referrer'    => 'HTTP_REFERER',
        'uid'         => 'UNIQUE_ID',
    );

    public function __construct($serverData = null, ?array $extraFields = null)
    {
        parent::__construct($serverData, $extraFields);

        if (isset($this->serverData['HTTP_X_FORWARDED_FOR'])) {
            // If there is X_FORWARDED_FOR header, set Client IP as the first IP in the list
            $xff = explode(',', $this->serverData['HTTP_X_FORWARDED_FOR']);
            $this->serverData['REMOTE_ADDR'] = $xff[0] ?? $this->serverData['REMOTE_ADDR'];
        } else if (isset($this->serverData['TRUE-CLIENT-IP'])) {
            // Akamai Header
            $this->serverData['REMOTE_ADDR'] = $this->serverData['TRUE-CLIENT-IP'];
        }

        $this->serverData['UNIQUE_ID'] = $this->serverData['HTTP_X_UNIQUE_ID']
            ?? $this->serverData['UNIQUE_ID']
            ?? substr(
                hash('sha256', uniqid('', true)),
                0,
                self::DEFAULT_UNIQUE_ID_LENGTH
            );

        if (extension_loaded('newrelic')) {
            newrelic_add_custom_parameter('uid', $this->serverData['UNIQUE_ID']);
        }
    }
}
