<?php

namespace AlloCine\CronReporterBundle\Transports\Drivers;

use AlloCine\CronReporterBundle\Exception\TransportException;
use AlloCine\CronReporterBundle\Model\CronReporter;
use CL\Slack\Model\Attachment;
use CL\Slack\Model\AttachmentField;
use CL\Slack\Payload\ChatPostMessagePayload;
use CL\Slack\Transport\ApiClient;

class Slack extends AbstractTransport
{

    /**
     * @var ApiClient
     */
    private $client;

    /**
     * Slack constructor.
     *
     * @param ApiClient $apiClient
     */
    public function __construct(ApiClient $apiClient = null)
    {
        $this->client = $apiClient;
    }

    /**
     * @inheritdoc
     */
    public static function getName(): string
    {
        return 'slack';
    }

    /**
     * @inheritdoc
     */
    public function send(CronReporter $cronReporter, callable $onSuccess = null, callable $onFailure = null): bool
    {
        $cronReporterPayload = $cronReporter->getExtraPayload();
        $payload = new ChatPostMessagePayload();
        $payload->setChannel($this->configuration['target']);
        $payload->setUsername('slackbot');
        $payload->setIconEmoji(':trollface:');

        $extraData = [];
        foreach ($cronReporterPayload['data'] ?? [] as $type => $messages) {
            $extraData[$type] = implode("\n", $messages);
        }

        $attachmentColor = 'danger';
        $errorField = null;
        if (
            $cronReporter->getStatus() === CronReporter::STATUS_STARTED
            ||
            $cronReporter->getStatus() === CronReporter::STATUS_SUCCESS
        ) {
            $attachmentFallback = sprintf(
                "Cron Monitor Fallback: %s %s\nCommand: %s\nStatus: %s",
                $cronReporter->getProject(),
                $cronReporter->getEnvironment(),
                $cronReporter->getJobName(),
                $cronReporter->getStatus()
            );

            if ($cronReporter->getStatus() === CronReporter::STATUS_STARTED) {
                $attachmentColor = '#439FE0';
            }
            else {
                $attachmentColor = 'good';
            }
        }
        else {
            $attachmentFallback = sprintf(
                "Cron Monitor Fallback: %s %s\nCommand: %s\nStatus: %s\nOutput:\n%s \nTokens:\n%s\nFallback Reason:\n`%s`\n\n\n",
                $cronReporter->getProject(), $cronReporter->getEnvironment(),
                $cronReporter->getJobName(),
                $cronReporter->getStatus(),
                $cronReporterPayload['output'] ?? '',
                json_encode($cronReporterPayload['tokens'] ?? []),
                $cronReporterPayload['message'] ?? ''
            );
            $errorField = new AttachmentField();
            $errorField->setShort(false);
            $errorField->setTitle('Error');
            $errorField->setValue($cronReporterPayload['message'] ?? '');
        }

        $attachment = new Attachment();
        $attachment->setColor($attachmentColor);
        $attachment->setFallback($attachmentFallback);

        $projectField = new AttachmentField();
        $projectField->setShort(true);
        $projectField->setTitle('Project');
        $projectField->setValue($cronReporter->getProject());
        $attachment->addField($projectField);

        $commandField = new AttachmentField();
        $commandField->setShort(true);
        $commandField->setTitle('Command');
        $commandField->setValue($cronReporter->getJobName());
        $attachment->addField($commandField);

        $environmentField = new AttachmentField();
        $environmentField->setShort(true);
        $environmentField->setTitle('Environment');
        $environmentField->setValue($cronReporter->getEnvironment());
        $attachment->addField($environmentField);

        $statusField = new AttachmentField();
        $statusField->setShort(true);
        $statusField->setTitle('Status');
        $statusField->setValue($cronReporter->getStatus());
        $attachment->addField($statusField);

        if ($errorField) {
            $attachment->addField($errorField);
        }

        $payload->addAttachment($attachment);
        try {
            $response = $this->client->send($payload);
            if ($response->isOk() === false) {
                throw TransportException::failureException(sprintf("%s\n%s", $response->getError(),
                    $response->getErrorExplanation()));
            }

            if (!is_null($onSuccess)) {
                call_user_func($onSuccess, $cronReporter);
            }

            return true;
        } catch (\Exception $exception) {
            if (!is_null($onFailure)) {
                call_user_func($onFailure, $cronReporter);
            }

            return false;
        }
    }
}
