#!/usr/bin/env node
import * as optimist from "optimist";
import { exec } from "child_process";
import * as v5 from "uuid/v5";

import emit from "./api";
import { Status, AllocinePayload } from "./interfaces";

const UUID_NAMESPACE = "cdb8630e-cf8e-430f-bef6-3a3f1073772a";

// all options
const argv = optimist
  .usage("Listen all event for a command")
  .alias("e", "env")
  .describe("e", "Env ( staging | dev | prod )")
  .demand("p")
  .alias("p", "project")
  .describe("p", "Project name")
  .demand("d")
  .alias("d", "description")
  .describe("d", "Description")
  .demand("j")
  .alias("j", "jobName")
  .describe("j", "Job name")
  .demand("c")
  .alias("c", "command")
  .describe("c", "Command").argv;

const getMemoryUsage = () => {
  return process.memoryUsage().heapUsed;
};

const getDateYmdHisu = () => {
  const d = new Date();
  return `${d.getFullYear()}${d.getMonth()}${d.getDay()}${d.getHours()}${d.getMinutes()}${d.getSeconds()}${d.getMilliseconds()}`;
};

const start = async () => {
  const startTime = new Date();
  const environments = process.env;

  let status: Status = "started";

  const environment = argv.env || "staging";
  const project = argv.project;
  const jobName = argv.jobName;

  const uuid = v5(
    `${environment}-${project}-${jobName}-${getDateYmdHisu()}`,
    UUID_NAMESPACE,
  );

  let payload: AllocinePayload = {
    brand: null,
    uuid,
    description: argv.description,
    duration: 0,
    environment,
    jobName,
    project,
    status,
    extraPayload: {
      message: undefined,
      output: null,
      error_output: null,
      standard_output: null,
      memory_usage: getMemoryUsage(),
      tokens: {
        arguments: {},
        environments,
        options: { command: argv.command },
      },
    },
  };

  await emit(payload);

  try {
    const payloadSent: AllocinePayload = await new Promise(resolve => {
      const child = exec(argv.command, (err, stdout, stderr) => {
        payload.extraPayload.output = stdout;
        payload.extraPayload.standard_output = stdout;
        payload.duration =
          Math.abs(new Date().getTime() - startTime.getTime()) * 1000;

        if (err || stderr) {
          payload.status = "failed";
          payload.extraPayload.message = err || stderr;
          payload.extraPayload.error_output = `\nError:\n${err}\n\nStd error:\n${stderr}`;

          return resolve(payload);
        }

        payload.status = "success";
        return resolve(payload);
      });

      child.stdout.on("data", data => process.stdout.write(data));
      child.stderr.on("data", data => process.stdout.write(data));
    });

    await emit(payloadSent);
  } catch (err) {
    status = "failed";
    throw err;
  }
};

start();
