<?php

namespace Tests\Transports\Drivers;

use Monolog\Logger;
use WM\CronReporterBundle\Model\CronReporter;
use WM\CronReporterBundle\Transports\Drivers\Monolog;
use PHPUnit\Framework\TestCase;

class MonologTest extends TestCase
{

    /**
     * Test get name
     */
    public function testGetName()
    {
        $this->assertEquals('monolog', Monolog::getName());
    }

    /**
     * @dataProvider getDataWithoutCallbacks
     *
     * @param CronReporter  $cronReporter
     * @param callable|null $onSuccess
     * @param callable|null $onFailure
     */
    public function testSendWithoutCallbacks(
        CronReporter $cronReporter,
        callable $onSuccess = null,
        callable $onFailure = null
    ) {
        $loggerMock = $this->getLogger();
        $loggerMock->expects($this->any())
            ->method('info')
            ->withAnyParameters();

        $object = new Monolog($loggerMock);
        $object->setConfiguration(['transport' => Monolog::getName()]);
        $this->assertTrue($object->send($cronReporter, $onSuccess, $onFailure));
        @unlink(sprintf('/tmp/%s.%s.%s.txt', $cronReporter->getProject(), $cronReporter->getEnvironment(), $cronReporter->getJobName()));
    }

    /**
     * @dataProvider getDataWithCallbacks
     *
     * @param CronReporter $cronReporter
     * @param callable     $onSuccess
     * @param callable     $onFailure
     */
    public function testSendWithCallbacks(CronReporter $cronReporter, callable $onSuccess, callable $onFailure)
    {
        $loggerMock = $this->getLogger();
        $loggerMock->expects($this->any())
            ->method('info')
            ->withAnyParameters();

        $object = new Monolog($loggerMock);
        $object->setConfiguration(['transport' => Monolog::getName()]);
        $this->assertTrue($object->send($cronReporter, $onSuccess, $onFailure));
        $this->assertFileExists('/tmp/success.txt');
        $this->assertFileNotExists('/tmp/failure.txt');
        @unlink('/tmp/success.txt');
        @unlink('/tmp/failure.txt');
    }

    /**
     * @return array
     */
    public function getDataWithoutCallbacks()
    {
        return [
            [
                new CronReporter('phpunit', 'verification', 'test'),
                null,
                null,
            ],
        ];
    }

    /**
     * @return array
     */
    public function getDataWithCallbacks()
    {
        return [
            [
                new CronReporter('phpunit', 'verification', 'test'),
                function (CronReporter $cronReporter) {
                    file_put_contents('/tmp/success.txt', 'success');
                },
                function (CronReporter $cronReporter) {
                    file_put_contents('/tmp/failure.txt', 'failure');
                },
            ],
        ];
    }

    /**
     * @return \PHPUnit_Framework_MockObject_MockObject
     */
    private function getLogger()
    {
        return $this->getMockBuilder(Logger::class)
            ->disableOriginalConstructor()
            ->getMock();
    }

}
