<?php

namespace AlloCine\CronReporterBundle\EventListener;

use AlloCine\CronReporterBundle\Component\Console\CronReporterOutput;
use AlloCine\CronReporterBundle\Component\Console\CronReporterStreamOutput;
use AlloCine\CronReporterBundle\Traits\AutomaticCronReporter;
use Symfony\Component\Console\ConsoleEvents;
use Symfony\Component\Console\Event\ConsoleCommandEvent;
use Symfony\Component\Console\Event\ConsoleExceptionEvent;
use Symfony\Component\Console\Event\ConsoleTerminateEvent;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CronReporterEventSubscriber implements EventSubscriberInterface
{

    /**
     * @var array
     */
    const EXCLUDED_ARGUMENTS = [
        'command',
    ];

    /**
     * @var array
     */
    const EXCLUDED_OPTIONS = [
        'ansi',
        'env',
        'help',
        'no-ansi',
        'no-debug',
        'no-interaction',
        'no-optional-warmers',
        'no-warmup',
        'quiet',
        'verbose',
        'version',
    ];

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            ConsoleEvents::COMMAND => 'onConsoleBeforeStart',
            ConsoleEvents::EXCEPTION => 'onConsoleException',
            ConsoleEvents::TERMINATE => 'onConsoleTerminate',
        ];
    }

    /**
     * @param ConsoleCommandEvent $event
     */
    public function onConsoleBeforeStart(ConsoleCommandEvent $event)
    {
        $command = $event->getCommand();
        $reflection = new \ReflectionClass($command);
        $traits = $reflection->getTraits();
        if (!array_key_exists(AutomaticCronReporter::class, $traits)) {
            return;
        }
        $reflection = null;
        /** @var AutomaticCronReporter $command */
        $command->start($this->computeTokens($event->getInput()));
    }

    /**
     * @param ConsoleExceptionEvent $event
     */
    public function onConsoleException(ConsoleExceptionEvent $event)
    {
        $command = $event->getCommand();
        $reflection = new \ReflectionClass($command);
        $traits = $reflection->getTraits();
        if (!array_key_exists(AutomaticCronReporter::class, $traits)) {
            return;
        }
        $reflection = null;
        $exception = $event->getException();
        $output = $event->getOutput();
        $extraInformation = [];
        if ($output instanceof CronReporterOutput) {
            $extraInformation['output'] = $output->getBuffer();
        }
        $errorOutput = $output->getErrorOutput();
        if ($errorOutput instanceof CronReporterStreamOutput) {
            $extraInformation['error_output'] = $errorOutput->getBuffer();
        }
        $exceptionData = [];
        if (method_exists($exception, 'getData')) {
            $exceptionData = $exception->getData();
        }
        /** @var AutomaticCronReporter $command */
        $command->failure(array_merge([
            'trace' => $exception->getTraceAsString(),
            'message' => $exception->getMessage(),
            'data' => $exceptionData,
       ], $extraInformation));
    }

    /**
     * @param ConsoleTerminateEvent $event
     */
    public function onConsoleTerminate(ConsoleTerminateEvent $event)
    {
        $command = $event->getCommand();
        $reflection = new \ReflectionClass($command);
        $traits = $reflection->getTraits();
        if (!array_key_exists(AutomaticCronReporter::class, $traits)) {
            return;
        }
        $reflection = null;
        $output = $event->getOutput();
        $extraInformation = [];
        if ($output instanceof CronReporterOutput) {
            $extraInformation['output'] = $output->getBuffer();
        }
        $errorOutput = $output->getErrorOutput();
        if ($errorOutput instanceof CronReporterStreamOutput) {
            $extraInformation['error_output'] = $errorOutput->getBuffer();
        }
        /** @var AutomaticCronReporter $command */
        $command->end($extraInformation);
    }

    /**
     * @param InputInterface $input
     *
     * @return array
     */
    private function computeTokens(InputInterface $input): array
    {
        $tokens = ['arguments' => [], 'options' => []];

        foreach ($input->getArguments() as $name => $value) {
            if (!in_array($name, self::EXCLUDED_ARGUMENTS) && !is_null($value) && !empty($value)) {
                $tokens['arguments'][$name] = $value;
            }
        }
        foreach ($input->getOptions() as $name => $value) {
            if (!in_array($name, self::EXCLUDED_OPTIONS) && !is_null($value) && !empty($value)) {
                $tokens['options'][$name] = $value;
            }
        }

        return ['tokens' => $tokens];
    }
}
