<?php

namespace WM\CronReporterBundle\Model;

use Symfony\Component\Serializer\Annotation\Groups;

class CronReporter
{

    public const STATUS_STARTED = 'started';
    public const STATUS_FAILED = 'failed';
    public const STATUS_SUCCESS = 'success';
    public const STATUS_LOCKED = 'locked';
    public const STATUS_CRITICAL = 'critical';

    /**
     * @var array
     */
    public static $statuses = [
        self::STATUS_STARTED => self::STATUS_STARTED,
        self::STATUS_SUCCESS => self::STATUS_SUCCESS,
        self::STATUS_FAILED => self::STATUS_FAILED,
        self::STATUS_LOCKED => self::STATUS_LOCKED,
        self::STATUS_CRITICAL => self::STATUS_CRITICAL,
    ];

    /**
     * @var string|null
     * @Groups({"display"})
     */
    protected $id;

    /**
     * @var string
     * @Groups({"display", "create"})
     */
    protected $project;

    /**
     * @var string
     * @Groups({"display", "create"})
     */
    protected $jobName;

    /**
     * @var string|null
     * @Groups({"display", "create"})
     */
    protected $description;

    /**
     * @var \DateTime|null
     * @Groups({"display"})
     */
    protected $createdAt;

    /**
     * @var string
     * @Groups({"display", "create"})
     */
    protected $status;

    /**
     * @var int
     * @Groups({"display", "create"})
     */
    protected $duration;

    /**
     * @var array
     * @Groups({"display", "create"})
     */
    protected $extraPayload;

    /**
     * @var string
     * @Groups({"display", "create"})
     */
    protected $environment;

    /**
     * @var string|null
     * @Groups({"display", "create"})
     */
    protected $brand;

    /**
     * CronReporter constructor.
     *
     * @param string      $project
     * @param string      $jobName
     * @param string      $environment
     * @param null|string $brand
     * @param string|null $description
     * @throws \Exception
     */
    public function __construct(
        string $project,
        string $jobName,
        string $environment,
        string $brand = null,
        string $description = null
    ) {
        $this->setProject($project);
        $this->setJobName($jobName);
        $this->setStatus(self::STATUS_STARTED);
        $this->setCreatedAt(new \DateTime());
        $this->setDuration(0);
        $this->setExtraPayload([]);
        $this->setEnvironment($environment);
        $this->setBrand($brand);
        $this->setDescription($description);
    }

    /**
     * @param int $exitCode
     * @return string
     */
    public static function mapExitCodeToStatus(int $exitCode)
    {
        switch ($exitCode) {
            case 0:
                return self::STATUS_SUCCESS;
            case 1:
                return self::STATUS_FAILED;
            case -1:
                return self::STATUS_LOCKED;
            case 65: # data format error
                return self::STATUS_CRITICAL;
            default:
                return self::STATUS_FAILED;
        }
    }

    /**
     * @return null|string
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param null|string $id
     *
     * @return self
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * @return string
     */
    public function getProject(): string
    {
        return $this->project;
    }

    /**
     * @param string $project
     *
     * @return self
     */
    public function setProject(string $project): CronReporter
    {
        $this->project = $project;

        return $this;
    }

    /**
     * @return string
     */
    public function getJobName(): string
    {
        return $this->jobName;
    }

    /**
     * @param string $jobName
     *
     * @return self
     */
    public function setJobName(string $jobName): CronReporter
    {
        $this->jobName = $jobName;

        return $this;
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @param string $description
     *
     * @return self
     */
    public function setDescription(string $description = null): CronReporter
    {
        $this->description = $description;

        return $this;
    }

    /**
     * @return \DateTime|null
     */
    public function getCreatedAt()
    {
        return $this->createdAt;
    }

    /**
     * @param \DateTime|null $createdAt
     *
     * @return self
     */
    public function setCreatedAt($createdAt)
    {
        $this->createdAt = $createdAt;

        return $this;
    }

    /**
     * @return string
     */
    public function getStatus(): string
    {
        return $this->status;
    }

    /**
     * @param string $status
     *
     * @return self
     * @throws \InvalidArgumentException
     */
    public function setStatus(string $status): CronReporter
    {
        if (!in_array($status, self::$statuses)) {
            throw new \InvalidArgumentException(sprintf('The status "%s" is not authorized. Authorized statuses: %s',
                $status, implode(',', self::$statuses)));
        }
        $this->status = $status;

        return $this;
    }

    /**
     * @return int
     */
    public function getDuration(): int
    {
        return $this->duration;
    }

    /**
     * @param int $duration
     *
     * @return self
     */
    public function setDuration(int $duration): CronReporter
    {
        $this->duration = $duration;

        return $this;
    }

    /**
     * @return array
     */
    public function getExtraPayload(): array
    {
        return $this->extraPayload;
    }

    /**
     * @param array $extraPayload
     *
     * @return self
     */
    public function setExtraPayload(array $extraPayload): CronReporter
    {
        $this->extraPayload = $extraPayload;

        return $this;
    }

    /**
     * @param array $extraPayload
     *
     * @return CronReporter
     */
    public function addExtraPayload(array $extraPayload): CronReporter
    {
        $this->extraPayload = array_merge($this->extraPayload, $extraPayload);

        return $this;
    }

    /**
     * @return string
     */
    public function getEnvironment(): string
    {
        return $this->environment;
    }

    /**
     * @param string $environment
     *
     * @return self
     */
    public function setEnvironment(string $environment): CronReporter
    {
        $this->environment = $environment;

        return $this;
    }

    /**
     * @return null|string
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @param null|string $brand
     *
     * @return self
     */
    public function setBrand(string $brand = null)
    {
        $this->brand = $brand;

        return $this;
    }
}
