<?php

namespace WM\CronReporterBundle\EventListener;

use function in_array;
use WM\CronReporterBundle\Component\Console\CronReporterOutput;
use WM\CronReporterBundle\Component\Console\CronReporterStreamOutput;
use WM\CronReporterBundle\Component\Console\SharedBufferCronReporterOutput;
use WM\CronReporterBundle\Model\CronReporter;
use WM\CronReporterBundle\Traits\AutomaticCronReporter;
use Symfony\Component\Console\ConsoleEvents;
use Symfony\Component\Console\Event\ConsoleCommandEvent;
use Symfony\Component\Console\Event\ConsoleErrorEvent;
use Symfony\Component\Console\Event\ConsoleTerminateEvent;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CronReporterEventSubscriber implements EventSubscriberInterface
{

    /**
     * @var array
     */
    const EXCLUDED_ARGUMENTS = [
        'command',
    ];

    /**
     * @var array
     */
    const EXCLUDED_OPTIONS = [
        'ansi',
        'env',
        'help',
        'no-ansi',
        'no-debug',
        'no-interaction',
        'no-optional-warmers',
        'no-warmup',
        'quiet',
        'verbose',
        'version',
    ];

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            ConsoleEvents::COMMAND => 'onConsoleBeforeStart',
            ConsoleEvents::ERROR => 'onConsoleError',
            ConsoleEvents::TERMINATE => 'onConsoleTerminate',
        ];
    }

    /**
     * @param ConsoleCommandEvent $event
     * @throws \ReflectionException
     */
    public function onConsoleBeforeStart(ConsoleCommandEvent $event)
    {
        $command = $event->getCommand();
        if (!$command) {
            return;
        }
        $reflection = new \ReflectionClass($command);
        $traits = $reflection->getTraits();
        if (!array_key_exists(AutomaticCronReporter::class, $traits)) {
            return;
        }
        $reflection = null;
        /** @var AutomaticCronReporter $command */
        $command->start($this->computeTokens($event->getInput()));
    }

    /**
     * @param ConsoleErrorEvent $event
     * @throws \ReflectionException
     */
    public function onConsoleError(ConsoleErrorEvent $event)
    {
        $command = $event->getCommand();
        if (!$command) {
            return;
        }
        $reflection = new \ReflectionClass($command);
        $traits = $reflection->getTraits();
        if (!array_key_exists(AutomaticCronReporter::class, $traits)) {
            return;
        }
        $reflection = null;
        $exception = $event->getError();
        $extraInformation = $this->computeExtraInformation($event->getOutput());
        $exceptionData = [];
        if (method_exists($exception, 'getData')) {
            $exceptionData = $exception->getData();
        }
        /** @var AutomaticCronReporter $command */
        $command->failure(array_merge([
            'trace' => $exception->getTraceAsString(),
            'message' => $exception->getMessage(),
            'data' => $exceptionData,
        ], $extraInformation));
    }

    /**
     * @param ConsoleTerminateEvent $event
     * @throws \ReflectionException
     */
    public function onConsoleTerminate(ConsoleTerminateEvent $event)
    {
        $command = $event->getCommand();
        if (!$command) {
            return;
        }
        $reflection = new \ReflectionClass($command);
        $traits = $reflection->getTraits();
        if (!array_key_exists(AutomaticCronReporter::class, $traits)) {
            return;
        }
        $reflection = null;
        $extraInformation = $this->computeExtraInformation($event->getOutput());
        $status = CronReporter::mapExitCodeToStatus($event->getExitCode());
        /** @var AutomaticCronReporter $command */
        $command->end($status, $extraInformation);
    }

    /**
     * @param InputInterface $input
     *
     * @return array
     */
    private function computeTokens(InputInterface $input): array
    {
        $tokens = ['environments' => [], 'arguments' => [], 'options' => []];

        $tokens['environments'] = $_SERVER;
        foreach ($input->getArguments() as $name => $value) {
            if (!in_array($name, self::EXCLUDED_ARGUMENTS) && !empty($value)) {
                $tokens['arguments'][$name] = $value;
            }
        }
        foreach ($input->getOptions() as $name => $value) {
            if (!in_array($name, self::EXCLUDED_OPTIONS) && !empty($value)) {
                $tokens['options'][$name] = $value;
            }
        }

        return ['tokens' => $tokens];
    }

    /**
     * @param OutputInterface $output
     *
     * @return array
     */
    private function computeExtraInformation(OutputInterface $output): array
    {
        $extraInformation = [];
        if ($output instanceof CronReporterOutput) {
            $extraInformation['standard_output'] = $output->getBuffer();
            $extraInformation['output'] = $output->getBuffer();
        }
        $errorOutput = $output->getErrorOutput();
        if ($errorOutput instanceof CronReporterStreamOutput) {
            $extraInformation['error_output'] = $errorOutput->getBuffer();
        }
        if ($output instanceof SharedBufferCronReporterOutput) {
            $extraInformation['output'] = $output->getSharedBuffer();
        }

        return $extraInformation;
    }
}
