<?php

namespace AlloCine\CronReporterBundle\Transports\Drivers;

use AlloCine\CronReporterBundle\Exception\TransportException;
use AlloCine\CronReporterBundle\Model\CronReporter;
use CL\Slack\Model\Attachment;
use CL\Slack\Model\AttachmentField;
use CL\Slack\Payload\ChatPostMessagePayload;
use CL\Slack\Transport\ApiClient;

class Slack extends AbstractTransport
{

    /**
     * @var ApiClient
     */
    private $client;

    /**
     * Slack constructor.
     *
     * @param ApiClient $apiClient
     */
    public function __construct(ApiClient $apiClient = null)
    {
        $this->client = $apiClient;
    }

    /**
     * @inheritdoc
     */
    public static function getName(): string
    {
        return 'slack';
    }

    /**
     * @inheritdoc
     */
    public function send(CronReporter $cronReporter, callable $onSuccess = null, callable $onFailure = null): bool
    {
        $cronReporterPayload = $cronReporter->getExtraPayload();
        $payload = new ChatPostMessagePayload();
        $payload->setChannel($this->configuration['target']);
        $payload->setUsername('slackbot');
        $payload->setIconEmoji(':trollface:');

        $extraData = [];
        foreach ($cronReporterPayload['data'] ?? [] as $type => $messages) {
            $extraData[$type] = implode("\n", $messages);
        }

        $attachmentColor = 'danger';
        if (
            $cronReporter->getStatus() === CronReporter::STATUS_STARTED
            ||
            $cronReporter->getStatus() === CronReporter::STATUS_SUCCESS
        ) {
            $attachmentFallback = sprintf(
                "Cron Monitor Fallback: %s %s\nCommand: %s\nStatus: %s",
                $cronReporter->getProject(),
                $cronReporter->getEnvironment(),
                $cronReporter->getJobName(),
                $cronReporter->getStatus()
            );
            $attachmentColor = 'good';
        }
        else {
            $attachmentFallback = sprintf(
                "Cron Monitor Fallback: %s %s\nCommand: %s\nStatus: %s\nOutput:\n%s \nTokens:\n%s\nFallback Reason:\n`%s`\n\n\n",
                $cronReporter->getProject(), $cronReporter->getEnvironment(),
                $cronReporter->getJobName(),
                $cronReporter->getStatus(),
                $cronReporterPayload['output'] ?? '',
                json_encode($cronReporterPayload['tokens'] ?? []),
                $cronReporterPayload['message'] ?? ''
            );
            $payload->setText(
                sprintf("```%s %s```",
                    $cronReporterPayload['output'] ?? '',
                    json_encode($cronReporterPayload['tokens'] ?? [])
                )
            );
        }

        $attachment = new Attachment();
        $attachment->setColor($attachmentColor);
        $attachment->setFallback($attachmentFallback);

        $projectField = new AttachmentField();
        $projectField->setShort(true);
        $projectField->setTitle('Project');
        $projectField->setValue($cronReporter->getProject());
        $attachment->addField($projectField);

        $environmentField = new AttachmentField();
        $environmentField->setShort(true);
        $environmentField->setTitle('Environment');
        $environmentField->setValue($cronReporter->getEnvironment());
        $attachment->addField($environmentField);

        $commandField = new AttachmentField();
        $commandField->setShort(true);
        $commandField->setTitle('Command');
        $commandField->setValue($cronReporter->getJobName());
        $attachment->addField($commandField);

        $fallbackReasonField = new AttachmentField();
        $fallbackReasonField->setShort(true);
        $fallbackReasonField->setTitle('Fallback Reason');
        $fallbackReasonField->setValue($cronReporterPayload['message'] ?? '');
        $attachment->addField($fallbackReasonField);

        try {
            $response = $this->client->send($payload);
            if ($response->isOk() === false) {
                throw TransportException::failureException(sprintf("%s\n%s", $response->getError(),
                    $response->getErrorExplanation()));
            }

            if (!is_null($onSuccess)) {
                call_user_func($onSuccess, $cronReporter);
            }

            return true;
        } catch (\Exception $exception) {
            if (!is_null($onFailure)) {
                call_user_func($onFailure, $cronReporter);
            }

            return false;
        }
    }
}
