<?php

namespace AlloCine\CronReporterBundle\Traits;

use AlloCine\CronReporterBundle\DependencyInjection\Configuration;
use AlloCine\CronReporterBundle\Model\CronReporter as Reporter;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\HttpFoundation\Request as RequestFoundation;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Stopwatch\Stopwatch;

trait CronReporter
{

    use ContainerAwareTrait;

    /**
     * Mandatory to get the command name
     *
     * @inheritdoc
     */
    abstract public function getName();

    /**
     * @var bool
     */
    private $_initialized = false;

    /**
     * @var bool
     */
    private $_enabled;

    /**
     * @var null|Client
     */
    private static $_client;

    /**
     * @var null|SerializerInterface
     */
    private static $_serializer;

    /**
     * @var array
     */
    private static $_configuration;

    /**
     * @var Reporter
     */
    private $_reporter;

    /**
     * @var Stopwatch
     */
    private $_stopWatch;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @required
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger): void
    {
        $this->logger = $logger;
    }

    /**
     * @return LoggerInterface
     */
    public function getLogger(): LoggerInterface
    {
        // TODO remove this when all projects use this Trait pass to use setLogger() to inject the logger
        if (null === $this->logger) {
            return $this->container->get('logger');
        }
        return $this->logger;
    }

    /**
     * @param array $extraInformation
     *
     * @return Reporter|null
     */
    public function start(array $extraInformation = [])
    {
        $bundleAlias = Configuration::getRootName();
        $this->_enabled = (bool)$this->container->getParameter(sprintf('%s.enabled', $bundleAlias));
        if ($this->_enabled === false) {
            return null;
        }
        $this->initializeReporter($bundleAlias);
        $this->_stopWatch->start('cron');
        $project = $this->container->getParameter(sprintf('%s.project', $bundleAlias));
        $this->_reporter = new Reporter(
            $project['name'],
            $this->getName(),
            $this->container->getParameter('kernel.environment'),
            $project['brand'],
            $this->getDescription()
        );
        $this->_reporter->addExtraPayload($extraInformation);
        $this->callApi();

        return $this->_reporter;
    }

    /**
     * @param string $status
     * @param array  $extraInformation
     *
     * @return Reporter|null
     */
    public function end(string $status = Reporter::STATUS_SUCCESS, array $extraInformation = [])
    {
        if ($this->_enabled === false) {
            return null;
        }
        if (!$this->_initialized) {
            throw new \RuntimeException('You must call start first');
        }
        if (!$this->_stopWatch->isStarted('cron')) {
            return $this->_reporter;
        }
        $event = $this->_stopWatch->stop('cron');
        $this->_reporter->setDuration($event->getDuration() * 1000);
        $this->_reporter->setStatus($status);
        $this->_reporter->addExtraPayload(array_merge(['memory_usage' => $event->getMemory()], $extraInformation));
        $this->callApi();

        return $this->_reporter;
    }

    /**
     * @param array $extraInformation
     *
     * @return null|Reporter
     * @throws \RuntimeException
     */
    public function failure(array $extraInformation = [])
    {
        if ($this->_enabled === false) {
            return null;
        }
        if (!$this->_initialized) {
            throw new \RuntimeException('You must call start first');
        }
        if (!$this->_stopWatch->isStarted('cron')) {
            return $this->_reporter;
        }
        $event = $this->_stopWatch->stop('cron');
        $this->_reporter->setDuration($event->getDuration() * 1000);
        $this->_reporter->setStatus(Reporter::STATUS_FAILED);
        $this->_reporter->addExtraPayload(array_merge(['memory_usage' => $event->getMemory()], $extraInformation));
        $this->callApi();

        return $this->_reporter;
    }

    /**
     * @return Reporter|null
     * @throws \RuntimeException
     */
    public function getReporter()
    {
        if ($this->_enabled === false) {
            return null;
        }
        if (is_null($this->_reporter)) {
            throw new \RuntimeException('You must call start method');
        }

        return $this->_reporter;
    }

    /**
     * Call API
     */
    private function callApi()
    {
        if (is_null($this->_reporter->getId())) {
            $verb = RequestFoundation::METHOD_POST;
            $route = sprintf('/%s/%s/cron-reporter', self::$_configuration['path'], self::$_configuration['version']);
        } else {
            $verb = RequestFoundation::METHOD_PUT;
            $route = sprintf('/%s/%s/cron-reporter/%s', self::$_configuration['path'], self::$_configuration['version'],
                $this->_reporter->getId());
        }
        $body = self::$_serializer->serialize(['allocine_cron_monitor_form_type_cron_reporter' => $this->_reporter],
            'json', ['groups' => ['create']]);
        $request = new Request($verb, $route, ['Content-Type' => 'application/json'], $body);

        try {
            $promise = self::$_client->sendAsync($request);
            $promise->then(
                function (ResponseInterface $response) {
                    try {
                        $this->_reporter = self::$_serializer->deserialize($response->getBody(),
                            Reporter::class, 'json',
                            ['groups' => ['display']]);
                    } catch (\Exception $exception) {
                        $this->getLogger()->critical($exception->getMessage(), ['caller' => __CLASS__]);
                    }
                },
                function (RequestException $exception) {
                    $this->getLogger()->critical($exception->getMessage(), ['caller' => __CLASS__]);
                }
            );
            $promise->wait();
        } catch (\Exception $exception) {
            $logger = $this->getLogger();
            $logger->error($exception->getMessage(), ['caller' => __CLASS__]);
            $transport = $this->container->getParameter('allo_cine_cron_reporter.fallback')['transport'];

            $this->_reporter->addExtraPayload([
                'fallback_trace' => $exception->getTraceAsString(),
                'fallback_message' => $exception->getMessage(),
            ]);
            $this->container->get('allocine_cron_reporter.transports.transport_factory')
                ->get($transport)
                ->send($this->_reporter,
                    function (Reporter $cronReporter) {
                        $cronReporter->addExtraPayload(['trace' => null, 'message' => null]);
                    },
                    function (Reporter $cronReporter) use ($logger) {
                        $logger->critical(sprintf('Fail to fallback for %s %s %s', $cronReporter->getProject(),
                            $cronReporter->getEnvironment(), $cronReporter->getJobName()));
                    }
                );
        }
    }

    /**
     * @param string $bundleAlias
     */
    private function initializeReporter(string $bundleAlias)
    {
        if (is_null(self::$_client) || is_null(self::$_serializer)) {
            self::$_configuration = $this->container->getParameter(sprintf('%s.api', $bundleAlias));
            self::$_client = new Client([
                'base_uri' => self::$_configuration['host'],
                'timeout' => self::$_configuration['timeout'],
            ]);
            self::$_serializer = $this->container->get('serializer');
            $this->_stopWatch = new Stopwatch();
            $this->_initialized = true;
        }
    }
}
