package cmd

import (
	"errors"
	"fmt"
	"log"
	"os"
	"os/exec"
	"strings"
	"time"

	"git.webedia-group.net/allocine/cron-reporter-bundle/cli/api"
	"github.com/spf13/cobra"
)

var rootCmdOpts struct {
	noANSI      bool
	project     string
	job         string
	environment string
	description string
	rawCommand  string
	command     string
	commandArgs []string
}

// CreateRootCmd creates the root command
func CreateRootCmd() *cobra.Command {
	cmd := &cobra.Command{
		Use:     "cron 'command to run'",
		Short:   "Cron reporter",
		RunE:    run,
		PreRunE: preRun,
	}

	cmd.PersistentFlags().StringVarP(&rootCmdOpts.project, "project", "p", "", "Project name")
	_ = cmd.MarkPersistentFlagRequired("project")

	cmd.PersistentFlags().StringVarP(&rootCmdOpts.description, "description", "d", "", "Description")

	cmd.PersistentFlags().StringVarP(&rootCmdOpts.job, "job", "j", "", "Job name")
	_ = cmd.MarkPersistentFlagRequired("job")

	cmd.PersistentFlags().StringVarP(&rootCmdOpts.environment, "environment", "e", "", "Environment")
	_ = cmd.MarkPersistentFlagRequired("environment")

	cmd.PersistentFlags().BoolVar(&rootCmdOpts.noANSI, "no-ansi", false, "Do not use ANSI color")

	return cmd
}

func preRun(cmd *cobra.Command, args []string) error {
	if len(args) < 1 || len(args[0]) == 0 {
		return errors.New("command to run must be specified")
	}
	rootCmdOpts.rawCommand = args[0]
	splitArg := strings.SplitN(rootCmdOpts.rawCommand, " ", 2)
	rootCmdOpts.command = splitArg[0]
	if len(splitArg) > 1 {
		rootCmdOpts.commandArgs = strings.Split(splitArg[1], " ")
	}
	return nil
}

func emit(payload *api.CronReporterPayload) {
	_, err := api.Call(payload)
	if err != nil {
		log.Printf("Error while calling cron reporter API: %s\n", err)
	}
}

func run(cmd *cobra.Command, args []string) error {
	fmt.Printf("Command to execute: %s %s\n", rootCmdOpts.command, rootCmdOpts.commandArgs)

	varEnv := os.Environ()

	// Init UUID
	log.Printf("UUID of this session: %s", api.Uuid(rootCmdOpts.environment, rootCmdOpts.project, rootCmdOpts.job, nil))

	payload := &api.CronReporterPayload{}

	payload.Root.Brand = ""
	payload.Root.Environment = rootCmdOpts.environment
	payload.Root.Project = rootCmdOpts.project
	payload.Root.JobName = rootCmdOpts.job
	payload.Root.Description = rootCmdOpts.description
	payload.Root.Duration = 0
	payload.Root.Status = api.StatusStarted
	payload.Root.ExtraPayload.Tokens.Environments = varEnv
	payload.Root.ExtraPayload.Tokens.Arguments = rootCmdOpts.commandArgs

	// Start emit
	emit(payload)

	c := exec.Command(rootCmdOpts.command, rootCmdOpts.commandArgs...)

	startTime := time.Now()
	out, err := c.Output()
	endTime := time.Now()

	payload.Root.ExtraPayload.MemoryUsage = getExecutedCmdMemoryUsage(c)

	fmt.Printf("Output:\n%s\n", string(out))
	if err != nil {
		fmt.Printf("Error:\n%s\n", err.Error())
	}

	payload.Root.Duration = int(endTime.Sub(startTime).Nanoseconds() / 1000000)
	payload.Root.ExtraPayload.Output = string(out)
	payload.Root.ExtraPayload.StandardOutput = string(out)

	payload.Root.Status = api.StatusSuccess
	if err != nil {
		payload.Root.Status = api.StatusFailed
		payload.Root.ExtraPayload.Message = err.Error()
		payload.Root.ExtraPayload.ErrorOutput = err.Error()
	}

	// End emit
	emit(payload)

	return nil
}
