package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"strconv"
	"time"

	"git.webedia-group.net/allocine/cron-reporter-bundle/cli/utils"
	"github.com/google/uuid"
)

const Endpoint = "https://deployment-internal.allocine.net/api/v1/cron-reporter/"
const StatusStarted = "started"
const StatusLocked = "locked"
const StatusSuccess = "success"
const StatusFailed = "failed"
const StatusCritical = "critical"

// NameSpace is the namespace used to calculate uuid
var NameSpace = uuid.Must(uuid.Parse("cdb8630e-cf8e-430f-bef6-3a3f1073772a"))
var runUuid *uuid.UUID

func Uuid(environment string, project string, job string, currentTime *time.Time) *uuid.UUID {
	if runUuid == nil {
		if currentTime == nil {
			nowTime := time.Now()
			currentTime = &nowTime
		}
		v := uuid.NewSHA1(NameSpace, []byte(fmt.Sprintf("%s-%s-%s-%s", environment, project, job, strconv.FormatInt(currentTime.UnixNano(), 10))))
		runUuid = &v
	}

	return runUuid
}

func Call(input *CronReporterPayload) (output *CronReporterPayload, err error) {
	client := &http.Client{}
	buffer, err := json.Marshal(input)
	if err != nil {
		return
	}
	//log.Printf("%s\n", buffer)

	url := Endpoint + runUuid.String()
	log.Printf("Calling %s\n", url)
	req, err := http.NewRequest(http.MethodPut, url, bytes.NewBuffer(buffer))
	if err != nil {
		err = fmt.Errorf("request error: %s", err)
		return
	}
	req.Header.Add("Content-Type", "application/json")
	req.Header.Add("User-Agent", fmt.Sprintf("cron-reporter-cli/%s-%s", utils.GetVersion(), utils.GetCommit()))

	res, err := client.Do(req)
	if err != nil {
		err = fmt.Errorf("request error: %s", err)
		return
	}
	defer res.Body.Close()

	var bodyBytes []byte
	if bodyBytes, err = ioutil.ReadAll(res.Body); err != nil {
		err = fmt.Errorf("send error, cannot read body: %d, %s", res.StatusCode, err)
		return
	}
	if res.StatusCode != 200 {
		err = fmt.Errorf("send error: %d, %s", res.StatusCode, string(bodyBytes))
		return
	} else {
		err = json.Unmarshal(bodyBytes, &output)
	}

	return
}
