<?php

namespace WM\CronReporterBundle\EventListener;

use function in_array;
use ReflectionClass;
use ReflectionException;
use Symfony\Component\Console\Command\Command;
use WM\CronReporterBundle\Component\Console\CronReporterOutput;
use WM\CronReporterBundle\Component\Console\CronReporterStreamOutput;
use WM\CronReporterBundle\Component\Console\SharedBufferCronReporterOutput;
use WM\CronReporterBundle\Model\CronReporter;
use WM\CronReporterBundle\Traits\AutomaticCronReporter;
use Symfony\Component\Console\ConsoleEvents;
use Symfony\Component\Console\Event\ConsoleCommandEvent;
use Symfony\Component\Console\Event\ConsoleErrorEvent;
use Symfony\Component\Console\Event\ConsoleTerminateEvent;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CronReporterEventSubscriber implements EventSubscriberInterface
{

    /**
     * @var array
     */
    const EXCLUDED_ARGUMENTS = [
        'command',
    ];

    /**
     * @var array
     */
    const EXCLUDED_OPTIONS = [
        'ansi',
        'env',
        'help',
        'no-ansi',
        'no-debug',
        'no-interaction',
        'no-optional-warmers',
        'no-warmup',
        'quiet',
        'verbose',
        'version',
    ];

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            ConsoleEvents::COMMAND => 'onConsoleBeforeStart',
            ConsoleEvents::ERROR => 'onConsoleError',
            ConsoleEvents::TERMINATE => 'onConsoleTerminate',
        ];
    }

    /**
     * @param ConsoleCommandEvent $event
     */
    public function onConsoleBeforeStart(ConsoleCommandEvent $event): void
    {
        $command = $event->getCommand();
        if (!$this->isAutomatic($command)) {
            return;
        }
        /** @var AutomaticCronReporter $command */
        $command->start($this->computeTokens($event->getInput()));
    }

    /**
     * @param ConsoleErrorEvent $event
     */
    public function onConsoleError(ConsoleErrorEvent $event): void
    {
        $command = $event->getCommand();
        if (!$this->isAutomatic($command)) {
            return;
        }
        /** @var AutomaticCronReporter $command */
        $exception = $event->getError();
        $extraInformation = $this->computeExtraInformation($event->getOutput());
        $exceptionData = [];
        if (method_exists($exception, 'getData')) {
            $exceptionData = $exception->getData();
        }
        $extraInformation = array_merge([
            'exception' => [
                'trace' => $exception->getTraceAsString(),
                'message' => $exception->getMessage(),
                'data' => $exceptionData,
            ],
        ], $extraInformation);
        if (!isset($extraInformation['error_output']) || empty($extraInformation['error_output'])) {
            $extraInformation['error_output'] = $exception->getMessage() . "\n" . $exception->getTraceAsString();
        }
        $command->failure($extraInformation);
    }

    /**
     * @param ConsoleTerminateEvent $event
     */
    public function onConsoleTerminate(ConsoleTerminateEvent $event): void
    {
        $command = $event->getCommand();
        if (!$this->isAutomatic($command)) {
            return;
        }
        /** @var AutomaticCronReporter $command */
        $extraInformation = $this->computeExtraInformation($event->getOutput());
        $status = CronReporter::mapExitCodeToStatus($event->getExitCode());
        $command->end($status, $extraInformation);
    }

    /**
     * @param InputInterface $input
     *
     * @return array
     */
    private function computeTokens(InputInterface $input): array
    {
        $tokens = ['environments' => [], 'arguments' => [], 'options' => []];

        $tokens['environments'] = $_SERVER;
        $tokens['environments']['host'] = gethostname() ?: '';
        foreach ($input->getArguments() as $name => $value) {
            if (!empty($value) && !in_array($name, self::EXCLUDED_ARGUMENTS)) {
                $tokens['arguments'][$name] = $value;
            }
        }
        foreach ($input->getOptions() as $name => $value) {
            if (!empty($value) && !in_array($name, self::EXCLUDED_OPTIONS)) {
                $tokens['options'][$name] = $value;
            }
        }

        return ['tokens' => $tokens];
    }

    /**
     * @param OutputInterface $output
     *
     * @return array
     */
    private function computeExtraInformation(OutputInterface $output): array
    {
        $extraInformation = [];
        if ($output instanceof CronReporterOutput) {
            $extraInformation['standard_output'] = $output->getBuffer();
        }
        $errorOutput = $output->getErrorOutput();
        if ($errorOutput instanceof CronReporterStreamOutput) {
            $extraInformation['error_output'] = $errorOutput->getBuffer();
        }
        if ($output instanceof SharedBufferCronReporterOutput) {
            $extraInformation['standard_output'] = $output->getSharedBuffer();
        }

        return $extraInformation;
    }

    /**
     * If in automatic mode
     * @param Command|null $command
     * @return bool
     */
    private function isAutomatic(?Command $command): bool
    {
        if (!$command) {
            return false;
        }
        try {
            $reflection = new ReflectionClass($command);
            $traits = $reflection->getTraits();
            $reflection = null;
            if (array_key_exists(AutomaticCronReporter::class, $traits)) {
                return true;
            }
        } catch (ReflectionException $e) {
        }
        return false;
    }
}
