<?php

namespace WM\CronReporterBundle\Traits;

use Psr\Log\LoggerAwareTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use WM\CronReporterBundle\DependencyInjection\Configuration;
use WM\CronReporterBundle\Model\CronReporter as Reporter;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\HttpFoundation\Request as RequestFoundation;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Stopwatch\Stopwatch;
use WM\CronReporterBundle\Transports\TransportFactory;

trait CronReporter
{

    use ContainerAwareTrait;
    use LoggerAwareTrait;

    /**
     * Mandatory to get the command name
     *
     * @inheritdoc
     */
    abstract public function getName();

    /**
     * @var bool
     */
    private $_initialized = false;

    /**
     * @var bool
     */
    private $_enabled;

    /**
     * @var null|Client
     */
    private static $_client;

    /**
     * @var null|SerializerInterface
     */
    private static $_serializer;

    /**
     * @var array
     */
    private static $_configuration;

    /**
     * @var string
     */
    private static $bundleAlias;

    /**
     * @var Reporter
     */
    private $_reporter;

    /**
     * @var Stopwatch
     */
    private $_stopWatch;


    /**
     * @required
     * @param ContainerInterface $container
     */
    public function setContainer(ContainerInterface $container = null)
    {
        $this->container = $container;
    }

    /**
     * @required
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @return LoggerInterface
     */
    public function getLogger(): LoggerInterface
    {
        // TODO remove this when all projects use this Trait pass to use setLogger() to inject the logger
        if (null === $this->logger) {
            return $this->container->get('logger');
        }
        return $this->logger;
    }

    /**
     * @param array $extraInformation
     *
     * @return Reporter|null
     */
    public function start(array $extraInformation = [])
    {
        self::$bundleAlias = Configuration::getRootName();
        $this->_enabled = (bool)$this->container->getParameter(sprintf('%s.enabled', self::$bundleAlias));
        if ($this->_enabled === false) {
            return null;
        }
        $this->initializeReporter(self::$bundleAlias);
        $this->_stopWatch->start('cron');
        $project = $this->getConfiguration(self::$bundleAlias, 'project');
        $this->_reporter = new Reporter(
            $project['name'],
            $this->getName(),
            $this->container->getParameter('kernel.environment'),
            $project['brand'],
            $this->getDescription()
        );
        $this->_reporter->addExtraPayload($extraInformation);
        $this->callApi();

        return $this->_reporter;
    }

    /**
     * @param string $status
     * @param array  $extraInformation
     *
     * @return Reporter|null
     */
    public function end(string $status = Reporter::STATUS_SUCCESS, array $extraInformation = [])
    {
        if ($this->_enabled === false || null === $this->_reporter) {
            return null;
        }
        if (!$this->_initialized) {
            throw new \RuntimeException('You must call start method');
        }
        if (!$this->_stopWatch->isStarted('cron')) {
            return $this->_reporter;
        }
        $event = $this->_stopWatch->stop('cron');
        $this->_reporter
            ->setDuration($event->getDuration() * 1000)
            ->setStatus($status)
            ->addExtraPayload(array_merge(['memory_usage' => $event->getMemory()], $extraInformation));

        $context = [
            'id' => $this->_reporter->getId(),
            'environment' => $this->_reporter->getEnvironment(),
            'project' => $this->_reporter->getProject(),
            'brand' => $this->_reporter->getBrand(),
            'status' => $this->_reporter->getStatus(),
            'duration' => $this->_reporter->getDuration(),
            'extra' => $this->_reporter->getExtraPayload(),
        ];
        $this->getLogger()->info($this->_reporter->getJobName(), $context);

        $this->callApi();

        return $this->_reporter;
    }

    /**
     * @param array $extraInformation
     *
     * @return null|Reporter
     * @throws \RuntimeException
     */
    public function failure(array $extraInformation = [])
    {
        if ($this->_enabled === false || null === $this->_reporter) {
            return null;
        }
        if (!$this->_initialized) {
            throw new \RuntimeException('You must call start method');
        }
        if (!$this->_stopWatch->isStarted('cron')) {
            return $this->_reporter;
        }
        $event = $this->_stopWatch->stop('cron');
        $this->_reporter
            ->setDuration($event->getDuration() * 1000)
            ->setStatus(Reporter::STATUS_FAILED)
            ->addExtraPayload(array_merge(['memory_usage' => $event->getMemory()], $extraInformation));
        $this->callApi();

        return $this->_reporter;
    }

    /**
     * @return Reporter|null
     * @throws \RuntimeException
     */
    public function getReporter()
    {
        if ($this->_enabled === false) {
            return null;
        }
        if (is_null($this->_reporter)) {
            throw new \RuntimeException('You must call start method');
        }

        return $this->_reporter;
    }

    /**
     * Call API
     */
    private function callApi()
    {
        $bundleAlias = Configuration::getRootName();
        $verb = RequestFoundation::METHOD_PUT;
        $route = sprintf('/%s/%s/cron-reporter/%s', self::$_configuration['path'], self::$_configuration['version'],
            $this->_reporter->getUuid());
        $body = self::$_serializer->serialize([$bundleAlias => $this->_reporter],
            'json', ['groups' => ['create']]);
        $request = new Request($verb, $route, ['Content-Type' => 'application/json'], $body);

        try {
            $promise = self::$_client->sendAsync($request);
            $promise->then(
                function (ResponseInterface $response) {
                    try {
                        $this->_reporter = self::$_serializer->deserialize($response->getBody(),
                            Reporter::class, 'json',
                            ['groups' => ['display']]);
                        if (null === $this->_reporter->getId()) {
                            $this->getLogger()->emergency('Something is not right, at this point CronReporter should have an ID', [
                                'CronReporter' => $this->_reporter
                            ]);
                        }
                    } catch (\Exception $exception) {
                        $this->getLogger()->critical($exception->getMessage(), ['caller' => __CLASS__]);
                    }
                },
                function (RequestException $exception) {
                    $this->getLogger()->critical($exception->getMessage(), ['caller' => __CLASS__]);
                }
            );
            $promise->wait();
        } catch (Exception $exception) {
            $transport = $this->container->getParameter(sprintf('%s.fallback', self::$bundleAlias))['transport'];

            $this->_reporter->addExtraPayload([
                'fallback_trace' => $exception->getTraceAsString(),
                'fallback_message' => $exception->getMessage(),
            ]);
            $this->container->get(TransportFactory::class)
                ->get($transport)
                ->send($this->_reporter,
                    function (Reporter $cronReporter) {
                        $cronReporter->addExtraPayload(['trace' => null, 'message' => null]);
                    },
                    function (Reporter $cronReporter) {
                        $this->getLogger()->critical(sprintf('Fail to fallback for %s %s %s', $cronReporter->getProject(),
                            $cronReporter->getEnvironment(), $cronReporter->getJobName()));
                    }
                );
        }
    }

    /**
     * @param string $bundleAlias
     */
    private function initializeReporter(string $bundleAlias)
    {
        if (is_null(self::$_client) || is_null(self::$_serializer)) {
            $project = $this->getConfiguration($bundleAlias, 'project');
            self::$_configuration = $this->container->getParameter(sprintf('%s.api', $bundleAlias));
            self::$_client = new Client([
                'base_uri' => self::$_configuration['host'],
                'timeout' => self::$_configuration['timeout'],
                'headers' => [
                    'User-Agent' => sprintf(
                        '%s/%s',
                        $project['name'],
                        $this->container->getParameter('kernel.environment')
                    )
                ]
            ]);
            self::$_serializer = $this->container->get('serializer');
            $this->_stopWatch = new Stopwatch();
            $this->_initialized = true;
        }
    }

    /**
     * @param string $bundleAlias
     * @param string $key
     * @return mixed
     */
    protected function getConfiguration(string $bundleAlias, string $key)
    {
        static $storage;

        if (!isset($storage)) {
            $storage = [];
        }

        if (!array_key_exists($key, $storage)) {
            $storage[$key] = $this->container->getParameter(sprintf(
                '%s.%s', $bundleAlias, $key
            ));
        }

        return $storage[$key];
    }
}
