<?php

namespace AlloCine\CronReporterBundle\Transports\Drivers;

use AlloCine\CronReporterBundle\Exception\TransportException;
use AlloCine\CronReporterBundle\Model\CronReporter;
use CL\Slack\Payload\ChatPostMessagePayload;
use CL\Slack\Transport\ApiClient;

class Slack extends AbstractTransport
{

    /**
     * @var ApiClient
     */
    private $client;

    /**
     * Slack constructor.
     *
     * @param ApiClient $apiClient
     */
    public function __construct(ApiClient $apiClient = null)
    {
        $this->client = $apiClient;
    }

    /**
     * @inheritdoc
     */
    public static function getName(): string
    {
        return 'slack';
    }

    /**
     * @inheritdoc
     */
    public function send(CronReporter $cronReporter, callable $onSuccess = null, callable $onFailure = null): bool
    {
        $cronReporterPayload = $cronReporter->getExtraPayload();
        $payload = new ChatPostMessagePayload();
        $payload->setChannel($this->configuration['target']);
        $payload->setUsername('slackbot');
        $payload->setIconEmoji(':trollface:');

        $extraData = [];
        foreach ($cronReporterPayload['data'] ?? [] as $type => $messages) {
            $extraData[$type] = implode("\n", $messages);
        }

        $payload->setText(sprintf("Cron Monitor Fallback: %s %s\nCommand: %s\nStatus: %s\nOutput:\n%s \nTokens:\n%s\nFallback Reason:\n`%s`\n\n\n",
                $cronReporter->getProject(), $cronReporter->getEnvironment(),
                $cronReporter->getJobName(),
                $cronReporter->getStatus(),
                $cronReporterPayload['output'] ?? '',
                json_encode($cronReporterPayload['tokens'] ?? []),
                $cronReporterPayload['message'] ?? '')
        );

        try {
            $response = $this->client->send($payload);
            if ($response->isOk() === false) {
                throw TransportException::failureException(sprintf("%s\n%s", $response->getError(),
                    $response->getErrorExplanation()));
            }

            if (!is_null($onSuccess)) {
                call_user_func($onSuccess, $cronReporter);
            }

            return true;
        } catch (\Exception $exception) {
            if (!is_null($onFailure)) {
                call_user_func($onFailure, $cronReporter);
            }

            return false;
        }
    }
}
