<?php

declare(strict_types=1);

namespace Webedia\UiBundle\Service;

use Symfony\Component\ExpressionLanguage\ExpressionLanguage;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class MenuUibService
{
    private array $config;
    private RequestStack $requestStack;
    private TranslatorInterface $translator;
    private UrlGeneratorInterface $urlGenerator;
    private ExpressionLanguage $expressionLanguage;

    public function __construct(
        array $headerUibConfig,
        RequestStack $requestStack,
        TranslatorInterface $translator,
        UrlGeneratorInterface $urlGenerator
    ) {
        $this->config = $headerUibConfig;
        $this->requestStack = $requestStack;
        $this->translator = $translator;
        $this->urlGenerator = $urlGenerator;
        $this->expressionLanguage = new ExpressionLanguage();
    }

    public function getMenu(bool $includeUrl = false): array
    {
        $request = $this->requestStack->getCurrentRequest();
        $universe = $request?->get('_universe') ?? '';

        return $this->processItems($this->config, $universe, $includeUrl);
    }

    private function processItems(array $items, string $universe, bool $includeUrl): array
    {
        $result = [];

        foreach ($items as $key => $item) {
            $routeName = $item['route_name'] ?? null;
            $routeParams = $item['route_params'] ?? [];
            $url = $item['url'] ?? null;
            $id = $item['id'] ?? null;
            $classes = $item['classes'] ?? [];

            $processed = [
                'key' => $key,
                'label' => $this->processLabel($item['label'] ?? ''),
                'route_name' => $routeName,
                'route_params' => $routeParams,
                'is_crawlable' => $this->evaluateCondition($item['is_crawlable'] ?? 'false', $universe),
                'is_current' => $this->evaluateCondition($item['is_current'] ?? 'false', $universe),
                'classes' => $classes,
            ];

            if ($id) {
                $processed['id'] = $id;
            }

            if ($url) {
                $processed['url'] = $url;
            }

            if ($includeUrl && $routeName) {
                try {
                    $processed['url'] = $this->urlGenerator->generate($routeName, $routeParams);
                } catch (\Exception $e) {
                    $processed['url'] = null;
                }
            }

            if (isset($item['items'])) {
                $processed['items'] = $this->processItems($item['items'], $universe, $includeUrl);
            }

            $result[$key] = $processed;
        }

        return $result;
    }

    private function processLabel(string $label): string
    {
        if (preg_match('/^"([^"]+)"\|trans\(\{\},\s*"([^"]+)"\)$/', $label, $matches)) {
            return $this->translator->trans($matches[1], [], $matches[2]);
        }

        if (preg_match('/^menu\.label\./', $label)) {
            return $this->translator->trans($label, [], 'menu');
        }

        return $label;
    }

    private function evaluateCondition(string $condition, string $universe): bool
    {
        if ($condition === 'true') {
            return true;
        }
        if ($condition === 'false') {
            return false;
        }

        try {
            $condition = preg_replace(
                '/app\.request\.get\(["\']_universe["\']\)/',
                '"' . $universe . '"',
                $condition
            );

            return (bool) $this->expressionLanguage->evaluate($condition);
        } catch (\Exception $e) {
            return false;
        }
    }
}

