import eventEmitter, { EventsTypes } from "../../services/events/eventEmitter";
import { on } from "../../dom/events";
import { removeClass, addClass, hasClass } from "../../dom/classes";
import debounce from "../../tools/debounce";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";
import { getOverlayState } from "../../module/overlay";
import {
  BODY,
  HEADER_MAIN,
  CLASS_MOBILE_SUBNAV_IS_OPEN,
  CLASS_BODY_HAS_BOTTOM_BAR,
  CLASS_BODY_BOTTOM_BAR_ON,
  CLASS_BODY_HEADER_MAIN_OFF
} from "../../constant/dom";

/**
 * Bottom Navigation Bar Module
 * Creates a mobile bottom navigation bar with configurable items
 */
let container = null;
let bottomBarConfig = null;
let itemMenu = null;
let bottomBarElements = [];

let isInitialized = false;
let headerMainIsVisible = true;
let bottomBarIsVisible = false;

/**
 * exemple of item parameters:
 * {
 *  name: "test",
 *  iconName: "video",
 *  label: "videos",
 *  link: "#",
 *  target: "_blank",
 *  rel: "noopener noreferrer",
 * }
 */

const defaultConfig = {
  items: [],
  containerClass: "bottom-bar-container",
  itemClass: "bottom-bar-item",
  labelClass: "bottom-bar-item-label",
  iconClass: "",
  activeClass: "active",
  scrollThreshold: 500,
  alwaysVisible: false,
  showOnLoad: false,
};

/**
 * Create the main container
 */
function createBottomBarContainer(bottomBarConfig) {
  container = document.createElement("nav");
  container.className = bottomBarConfig.containerClass;
}

/**
 * Create HTML for a single navigation item
 */
function createBottomBarItem(bottomBarConfigitem, item, index) {
  const isActive = isCurrentPage(item.link);
  const itemClasses = [
    bottomBarConfigitem.itemClass,
    ...(item.classes || []),
    ...(isActive ? [bottomBarConfigitem.activeClass] : []),
  ].join(" ");

  let bottomBarItem = null;

  if (item.link) {
    bottomBarItem = document.createElement("a");
    bottomBarItem.href = item.link;
  } else {
    bottomBarItem = document.createElement("span");
  }

  bottomBarItem.className = itemClasses;
  bottomBarItem.setAttribute("data-index", index);

  if (item.target) {
    bottomBarItem.setAttribute("target", item.target);
  }

  if (item.rel) {
    bottomBarItem.setAttribute("rel", item.rel);
  }

  if (item.iconName) {
    const iconItem = document.createElement("i");
    iconItem.className = `${bottomBarConfig.iconClass} ui-icons`;
    iconItem.textContent = item.iconNameActive && isActive ? item.iconNameActive : item.iconName;
    bottomBarItem.appendChild(iconItem);
  }

  if (item.label) {
    const labelItem = document.createElement("span");
    labelItem.className = bottomBarConfig.labelClass;
    labelItem.textContent = item.label;
    bottomBarItem.appendChild(labelItem);
  }

  if (item.name === "menu") {
    itemMenu = bottomBarItem;
  }

  return bottomBarItem;
}

/**
 * Render navigation items
 */
function injectItems(bottomBarConfig) {
    if (!bottomBarConfig.items || bottomBarConfig.items.length === 0) {
      console.warn("BottomBar: No items configured");
      return;
    }

    bottomBarConfig.items.map((item, index) => {
      const bottomBarItem = createBottomBarItem(bottomBarConfig, item, index);
      container.appendChild(bottomBarItem);
      bottomBarElements.push(bottomBarItem);
    });
}

/**
 * Check if the current page matches the item link
 */
function isCurrentPage(link) {
  const currentPath = window.location.pathname;
  const linkPath = new URL(link, window.location.origin).pathname;
  return currentPath === linkPath;
}

/**
 * Show the navigation
 */
function showBottomBar() {
  requestAnimationFrame(() => {
    addClass(BODY, CLASS_BODY_HEADER_MAIN_OFF);
    addClass(BODY, CLASS_BODY_BOTTOM_BAR_ON);
    headerMainIsVisible = false;
    bottomBarIsVisible = true;
  });
}

/**
 * Hide the navigation
 */
function showHeader() {
  requestAnimationFrame(() => {
    if (!bottomBarConfig.alwaysVisible) {
      removeClass(BODY, CLASS_BODY_BOTTOM_BAR_ON);
      bottomBarIsVisible = false;
    }
    removeClass(BODY, CLASS_BODY_HEADER_MAIN_OFF);

    // carefull bottomBarIsVisible doesn't reflect the actual state of the bottom bar,
    // since some BU could need to alway let the BB visible once it's been shown (cf. alwaysVisible config)
    // maybe we should use a headerIsVisible flag instead and invert the logic
    headerMainIsVisible = true;
  });
}

function handleScroll() {
  let oldScroll = window.pageYOffset || document.documentElement.scrollTop;

  const debouncedScroll = debounce(() => {
    // avoid hiding bottom bar when overlay is open
    // (issue linked on desktop where scroll bar disappear and let the content to fill the space and change the scroll position)
    if (getOverlayState().open) {
      return;
    }

    const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

    const scrollDelta = oldScroll - scrollTop;

    oldScroll = scrollTop;

    // avoid showing or made calculation to hide bottom bar when scrolling up or when scrollUp is less than 50px (and bottom bar is visible,
    // so it prevent also to do anything when bottomBar is visible and ou keep scrolling down due to the scrollDelta being negative in that case)
    // also prevent issue when closing overlay on mobile with menu showed (and bottom bar was not visible) that could trigger the showBottomBar()
    // (issue linked on desktop where scroll bar disappear and let the content to fill the space and change the scroll position)
    if ((scrollDelta >= 0 && headerMainIsVisible) || (!headerMainIsVisible && scrollDelta < 50)) {
      return;
    }

    if (headerMainIsVisible && scrollTop > bottomBarConfig.scrollThreshold) {
      showBottomBar();
    } else if (
      !headerMainIsVisible && (scrollTop <= bottomBarConfig.scrollThreshold || scrollDelta > 0)
    ) {
      showHeader();
    }

  }, 10);

  on(window, "scroll", debouncedScroll);
}

function injectBottomBar() {
  requestAnimationFrame(() => {
    BODY.appendChild(container);

    if (bottomBarConfig.showOnLoad) {
      addClass(BODY, CLASS_BODY_BOTTOM_BAR_ON);
      bottomBarIsVisible = true;
    }
  });

  // default event listener for menu item
  // shoudl be mandatory for all bottom bar items
  on(itemMenu, "click", () => {
    if (hasClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN)) {
      eventEmitter.emit(EventsTypes.MENU_CLOSE);
    } else {
      eventEmitter.emit(EventsTypes.MENU_OPEN);
    }
  });

  handleScroll();
}

function getBottomBarElements() {
  return {
    domElements: bottomBarElements,
    config: bottomBarConfig,
  };
}

export function getBottomBarState() {
  return { is_open: bottomBarIsVisible };
}

export default function initBottomBar(config = {}) {
  return new Promise((resolve) => {
    if (isInitialized || window.MqState.getState() > 1) {
      return;
    }

    addClass(BODY, CLASS_BODY_HAS_BOTTOM_BAR);

    bottomBarConfig = { ...defaultConfig, ...config };

    createBottomBarContainer(bottomBarConfig);
    injectItems(bottomBarConfig);
    injectBottomBar();

    isInitialized = true;

    resolve(getBottomBarElements());
  })
}
