import { on } from "../../dom/events";
import { addClass, removeClass, hasClass } from "../../dom/classes";
import scrollTo from "../../dom/scrollTo";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";
import loadYtPlayer from "../../module/player/yt";


let YT_MAIN_PLAYER_INSTANCE = null;
let CURRENT_VIDEO_ID = null;

const CURRENT_VIDEO_THUMB_CLASS = "current-video";
const MAIN_PLAYER = document.getElementById("player-main");
const PLAYER_CARDS = document.querySelectorAll(".video-card");

const VIDEO_IDS = [];

export default async function init(callback = null) {
    const ytPlayerData = await loadYtPlayer(onStateChange);

    YT_MAIN_PLAYER_INSTANCE = ytPlayerData.instance;
    CURRENT_VIDEO_ID = ytPlayerData.currentVideoId;

    PLAYER_CARDS.forEach((card, index, array) => {
        const videoId = card.dataset.videoId;

        if (hasClass(card, "video-card-shorts")) {
            return;
        }

        VIDEO_IDS.push(videoId);

        on(card, "click", (e) => {
            e.preventDefault();

            if (!videoId) {
                return false;
            }

            changeVideo(videoId, array);
        });
    });

    if (callback) {
        callback({
          currentVideoId: CURRENT_VIDEO_ID,
          playerInstance: YT_MAIN_PLAYER_INSTANCE,
          playerCards: PLAYER_CARDS,
        });
    }
}

function changeVideo(videoId, array) {
    // Check if player is ready before attempting to load video
    if (!YT_MAIN_PLAYER_INSTANCE || !YT_MAIN_PLAYER_INSTANCE.loadVideoById) {
        return false;
    }

    if (!CURRENT_VIDEO_ID || CURRENT_VIDEO_ID === videoId) {
        YT_MAIN_PLAYER_INSTANCE.playVideo();
        return;
    }

    removeCurrentCard();
    setCurrentCard(array, videoId);

    // Load and play the new video
    try {
        YT_MAIN_PLAYER_INSTANCE.loadVideoById(videoId);
        CURRENT_VIDEO_ID = videoId;
        // Note: playVideo() is called automatically after loadVideoById() completes
        // You can also add a small delay if needed:
        // setTimeout(() => YT_MAIN_PLAYER_INSTANCE.playVideo(), 100);
    } catch (error) {}
}

function setCurrentCard(array, videoId) {
    if (!array.length) {
        return;
    }

    array.forEach((item) => {
        if (item.dataset.videoId === videoId) {
            const currentLabel = document.createElement("span");
            currentLabel.className = "label label-type-current";
            currentLabel.textContent = "vidéo en cours";
            const CARD_THUMBNAIL = item.querySelector(".thumbnail");

            requestAnimationFrame(() => {
                CARD_THUMBNAIL.appendChild(currentLabel);
                addClass(item, "current-video");
            });
        }
    });

    scrollTo(MAIN_PLAYER, 100, true);
}

function removeCurrentCard() {
    const CURRENT_CARD = document.querySelectorAll(
        `.${CURRENT_VIDEO_THUMB_CLASS}`,
    );

    CURRENT_CARD.forEach((card) => {
        const CARD_THUMBNAIL = card.querySelector(".thumbnail");
        if (!CARD_THUMBNAIL) {
            return;
        }

        const LABEL = CARD_THUMBNAIL.querySelector(".label-type-current");

        requestAnimationFrame(() => {
            if (LABEL) {
                CARD_THUMBNAIL.removeChild(LABEL);
            }

            removeClass(card, CURRENT_VIDEO_THUMB_CLASS);
        });
    });
}

function onStateChange(event) {
    if (event.data === YT.PlayerState.ENDED) {
        const currentIndex = VIDEO_IDS.indexOf(CURRENT_VIDEO_ID);
        const nextVideoId =
            currentIndex < VIDEO_IDS.length - 1
                ? VIDEO_IDS[currentIndex + 1]
                : VIDEO_IDS[0];

        if (nextVideoId) {
            changeVideo(nextVideoId, PLAYER_CARDS);
        }
    }
}
