import eventEmitter, { EventsTypes } from "../../services/events/eventEmitter";
import { on } from "../../dom/events";
import { removeClass, addClass } from "../../dom/classes";
import debounce from "../../tools/debounce";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";

/**
 * Bottom Navigation Bar Module
 * Creates a mobile bottom navigation bar with configurable items
 */
let container = null;
let bottomBarConfig = null;
let itemMenu = null;
let bottomBarElements = [];

let isInitialized = false;
let bottomBarIsVisible = false;

const defaultConfig = {
    items: [],
    containerClass: "bottom-bar-container",
    itemClass: "bottom-bar-item",
    activeClass: "bottom-bar-item-active",
    labelClass: "bottom-bar-item-label",
    iconClass: "ui-icons",
    scrollThreshold: 500,
};

/**
 * Create the main container
 */
function createBottomBarContainer(bottomBarConfig) {
    container = document.createElement("nav");
    container.className = bottomBarConfig.containerClass;
}

/**
 * Create HTML for a single navigation item
 */
function createBottomBarItem(bottomBarConfigitem, item, index) {
    const isActive = isCurrentPage(item.link);
    const itemClasses = [
        bottomBarConfigitem.itemClass,
        ...(item.classes || []),
        ...(isActive ? [bottomBarConfigitem.activeClass] : []),
    ].join(" ");

    let bottomBarItem = null;

    if (item.link) {
        bottomBarItem = document.createElement("a");
        bottomBarItem.href = item.link;
    } else {
        bottomBarItem = document.createElement("span");
    }

    bottomBarItem.className = itemClasses;
    bottomBarItem.setAttribute("data-index", index);

    if (item.target) {
        bottomBarItem.setAttribute("target", item.target);
    }

    if (item.rel) {
        bottomBarItem.setAttribute("rel", item.rel);
    }

    if (item.iconName) {
        const iconItem = document.createElement("i");
        iconItem.className = `${bottomBarConfig.iconClass} ui-icons`;
        iconItem.textContent = item.iconName;
        bottomBarItem.appendChild(iconItem);
    }

    if (item.label) {
        const labelItem = document.createElement("span");
        labelItem.className = labelClass;
        labelItem.textContent = item.label;
        bottomBarItem.appendChild(labelItem);
    }

    if (item.name === "menu") {
        itemMenu = bottomBarItem;
    }

    return bottomBarItem;
}

/**
 * Render navigation items
 */
function injectItems(bottomBarConfig) {
    if (!bottomBarConfig.items || bottomBarConfig.items.length === 0) {
        console.warn("BottomBar: No items configured");
        return;
    }

    bottomBarConfig.items.map((item, index) => {
        const bottomBarItem = createBottomBarItem(bottomBarConfig, item, index);
        container.appendChild(bottomBarItem);
        bottomBarElements.push(bottomBarItem);
    });
}

/**
 * Check if the current page matches the item link
 */
function isCurrentPage(link) {
    const currentPath = window.location.pathname;
    const linkPath = new URL(link, window.location.origin).pathname;
    return currentPath === linkPath;
}

/**
 * Show the navigation
 */
function showBottomBar() {
    requestAnimationFrame(() => {
        removeClass(document.body, "bottom-bar-off");
        addClass(document.body, "bottom-bar-on");
        bottomBarIsVisible = true;
    });
}

/**
 * Hide the navigation
 */
function hideBottomBar() {
    requestAnimationFrame(() => {
        removeClass(document.body, "bottom-bar-on");
        addClass(document.body, "bottom-bar-off");
        bottomBarIsVisible = false;
    });
}

function handleScroll() {
    let oldScroll = window.pageYOffset || document.documentElement.scrollTop;

    const debouncedScroll = debounce(() => {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

        const scrollDelta = oldScroll - scrollTop;
        oldScroll = scrollTop;

        if (scrollDelta > 0 && !bottomBarIsVisible) {
            return;
        }

        if (!bottomBarIsVisible && scrollTop > bottomBarConfig.scrollThreshold) {
            showBottomBar();
        } else if (
            bottomBarIsVisible &&
            (scrollTop <= bottomBarConfig.scrollThreshold || scrollDelta > 0)
        ) {
            hideBottomBar();
        }

    }, 100);

    on(window, "scroll", debouncedScroll);
}

function injectBottomBar() {
    hideBottomBar();

    requestAnimationFrame(() => {
        document.body.appendChild(container);
    });

    // default event listener for menu item
    // shoudl be mandatory for all bottom bar items
    on(itemMenu, "click", () => {
        eventEmitter.emit(EventsTypes.MENU_OPEN);
    });

    handleScroll();
}

function getBottomBarElements() {
    return {
        domElements: bottomBarElements,
        config: bottomBarConfig,
    };
}

export function getBottomBarState() {
    return { is_open: bottomBarIsVisible };
}

export default function initBottomBar(config = {}) {
  return new Promise((resolve) => {
    if (isInitialized || window.MqState.getState() > 1) {
        return;
    }

    bottomBarConfig = { ...defaultConfig, ...config };

    createBottomBarContainer(bottomBarConfig);
    injectItems(bottomBarConfig);
    injectBottomBar();

    isInitialized = true;

    resolve(getBottomBarElements());
  })
}
