import { getConsentForAnalitycs } from './didomi';
import noop from '../tools/noop';

const PIXEL_URL = 'https://bpt.webedia-group.com/pixel.png?/';
const SITE_ROUTE =  window?._GLOBALS?.routes?.current;

export async function getConsentState(continueWoConsent = false) {
  return new Promise(resolve => {
    if (!window.__tcfapi) {
      resolve({status: false, code: 0});
      return;
    }

    const getConsent = (tcData) => {
      try {
        if (tcData.eventStatus === 'useractioncomplete') {
          // click on the CMP popin
          window.__tcfapi('removeEventListener', 2, getConsent);
          resolve({status: true, code: 1});
          return;
        } else if (tcData.eventStatus === 'tcloaded') {
          // has already clicked on the CMP popin
          window.__tcfapi('removeEventListener', 2, getConsent);
          resolve({status: true, code: 2});
          return;
        }
      } catch (e) {
        // eslint-disable-next-line no-console
        console.warn(`Error in getting consent : ${e}`);

        if (continueWoConsent) {
          resolve({status: false, code: 0});
        }
      }
    };

    window.__tcfapi('addEventListener', 2, getConsent);
  });
}

export async function triggerAfterConsent(timer = 250) {
  const consentState = await getConsentState(true);

  return new Promise(resolve => {
    if (consentState.status) {

      if (consentState.code !== 1) {
        // execute right away
        resolve(true);
        return;
      }

      // execute with a timer
      const delai = setTimeout(() => {
        resolve(true);

        if (delai) {
          clearTimeout(delai);
        }
      }, timer)

      return;
    }

    // execute anyway (error or no consent brand)
    resolve(false);
  });
}

export function trackConsent(hitCb = false, hitObj = false) {
  let tcfok = false;

  if (window.__tcfapi) {
    window.__tcfapi('addEventListener', 2, function (tcData) {
      try {
        switch (tcData.eventStatus) {
          case 'useractioncomplete':
            triggerConsentTracking(tcData.eventStatus, hitCb, hitObj);
            window.__tcfapi('removeEventListener', 2, noop, tcData.listenerId);
            break;
          case 'cmpuishown':
            triggerConsentTracking(tcData.eventStatus, false, false);
            break;
          case 'tcloaded':
            triggerConsentTracking(tcData.eventStatus, false, false);
            window.__tcfapi('removeEventListener', 2, noop, tcData.listenerId);
            break;
          default:
            triggerConsentTracking(tcData.eventStatus, hitCb, hitObj);
        }
      } catch (error) {
        triggerConsentTracking('tcferror', hitCb, hitObj);
      }
      tcfok = true;
    });
  }

  setTimeout(function () {
    if (!tcfok) {
      triggerConsentTracking('tcfnotloaded', false, false);
    }
  }, 2000);
}

async function triggerConsentTracking(tcfStatus, hitCb, hitObj) {
  let done = false;

  const trackingTimeout = setTimeout(() => {
    sendBecon('timeout', 'false', tcfStatus);
  }, 1000);

  function sendBecon(method, consent, tcfStatus) {
    if (!done) {
      if (hitCb) {
        hitCb(hitObj);
      }

      sendPixel(method, consent, tcfStatus);

      if (trackingTimeout) {
        clearTimeout(trackingTimeout);
      }

      done = true;
    }
  }

  const consent = await getConsentForAnalitycs(false);
  const consentToString = consent == undefined ? 'undefined' : consent;

  sendBecon('didomiOnReady', consentToString, tcfStatus);
}

function sendPixel(
  method,
  consent,
  status
) {
    if (!SITE_ROUTE) {
        return false;
    }
    const pixelParam = `domain=${window.location.host}&referrer=${document.referrer}&site_route=${SITE_ROUTE}&event=tcfapi&method=${method}&consent=${consent}&tcfStatus=${status}`;

    //console.debug('sendPixel : ', method, consent, status);

    try {
        fetch(PIXEL_URL + pixelParam);
    } catch (e) {
        // eslint-disable-next-line no-console
        console.warn(`Error sending pixel : ${e}`);
    }
}
