import eventEmitter, { EventsTypes } from "../../services/events/eventEmitter";
import { toggleClass, removeClass, addClass } from "../../dom/classes";
import { on } from "../../dom/events";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";
import request from "../../tools/request";
import {
  OVERLAY_ELEMENT,
  showOverlay,
  hideOverlay,
  getOverlayState,
} from "../../module/overlay";
import { BODY, CLASS_BODY_HEADER_HIDDEN } from "../../constant/dom";
import { buildVersionChanged } from  "../../tools/initBuild";

//Variables DOM
const HEADER_MAIN = document.getElementById("header-main");
const BTN_MOBILE_MENU = document.getElementById("header-main-menu-icon");
const BTN_CLOSE_MENU = document.getElementById("header-main-close-icon");
const NAV_MOBILE_PANEL = document.getElementById("header-nav-panel");
const NAV_ITEMS = document.querySelectorAll(".js-nav-item-holder");
const SUBNAV_ITEMS = document.querySelectorAll(".js-subnav-item");

const BTN_SUBNAV_MOBILE_TRIGGER = document.querySelectorAll(
  ".js-btn-expand-subnav"
);

const BTN_SEARCH_OPEN = document.getElementById("header-main-search-icon");
const BTN_SEARCH_CLOSE = document.getElementById("header-main-search-canceled");
const SEARCH_MOBILE_PANEL = document.getElementById("header-main-search-panel");

const CLASS_MOBILE_SUBNAV_IS_OPEN = "mobile-subnav-is-open";
const CLASS_MOBILE_SEARCH_IS_OPEN = "mobile-search-is-open";

let isSearchOpen = false;
let navObject = null;
let subnavInjected = false;

//Function header behavior on scroll
let windowTop =
  window.scrollY || window.pageYOffset || document.documentElement.scrollTop;

function headerSlider() {
  let HEADER_HIDDEN = false;

  on(window, "scroll", () => {
    if (window.scrollY < 79) {
      removeClass(BODY, CLASS_BODY_HEADER_HIDDEN);
      windowTop = window.scrollY;
      return false;
    }

    if (window.scrollY > windowTop && !HEADER_HIDDEN) {
      addClass(BODY, CLASS_BODY_HEADER_HIDDEN);
      HEADER_HIDDEN = true;
    }

    if (window.scrollY < windowTop && HEADER_HIDDEN) {
      removeClass(BODY, CLASS_BODY_HEADER_HIDDEN);
      HEADER_HIDDEN = false;
    }

    windowTop = window.scrollY;
  });
}

// Function show / hide Mobile menu
function toggleMobileMenu() {
  if (!BTN_MOBILE_MENU || !NAV_MOBILE_PANEL) {
    return false;
  }

  on(BTN_MOBILE_MENU, "click", () => {
    removeClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);
    toggleClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);

    // Set the control vars
    isSearchOpen = false;

    // check if overlay is on
    // if not we show overlay
    if (!getOverlayState().open) {
      showOverlay();
    }
  });

  on(BTN_CLOSE_MENU, "click", () => {
    toggleClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);

    // Set the control vars
    isSearchOpen = false;

    // check if overlay is on
    // if not we show overlay
    if (getOverlayState().open) {
      hideOverlay();
    }
  });
}

//Function show / hide mobile search
function toggleSearchMobilePanel() {
  if (!BTN_SEARCH_OPEN || !SEARCH_MOBILE_PANEL) {
    return false;
  }

  on(BTN_SEARCH_OPEN, "click", () => {
    removeClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);
    toggleClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);

    // check if overlay is on
    // if not we show overlay
    if (isSearchOpen) {
      hideOverlay();
    } else if (!getOverlayState().open) {
      showOverlay();
    }

    isSearchOpen = !isSearchOpen;
  });

  on(BTN_SEARCH_CLOSE, "click", () => {
    removeClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);
    hideOverlay();

    isSearchOpen = false;
  });
}

//Function submenu
function toggleMobileSubnavMenu() {
  if (!BTN_SUBNAV_MOBILE_TRIGGER.length || !SUBNAV_ITEMS.length) {
    return false;
  }

  let curent_item;

  BTN_SUBNAV_MOBILE_TRIGGER.forEach((item, index) => {
    on(item, "click", () => {
      const tmp_item = NAV_ITEMS[index];

      if (curent_item && curent_item != tmp_item) {
        removeClass(curent_item, "active");
      }
      toggleClass(tmp_item, "active");

      curent_item = tmp_item;
    });
  });
}

async function getSubnav() {
  const subnavXhrPath = NAV_MOBILE_PANEL?.getAttribute('data-subnav');
  if (subnavInjected || !subnavXhrPath) {  
    return false;
  }

  let menu = window.localStorage.getItem("nav") || null;

  if (!menu || buildVersionChanged()) {
    menu = await request(
      subnavXhrPath,
      { credentials: "same-origin" },
      false
    ).then((response) => response.text());

    window.localStorage.setItem("nav", menu);
  }

  navObject = JSON.parse(menu);
  renderSubnav();
  removeClass(NAV_MOBILE_PANEL, "header-nav-unloaded");

  subnavInjected = true;
}

//Function subnav items
function renderSubnav() {
  if (!navObject || !SUBNAV_ITEMS.length || subnavInjected) {
    return false;
  }

  requestAnimationFrame(() => {
    Object.entries(navObject).forEach(([navKey, navValues]) => {
      const tmpItem = Array.from(SUBNAV_ITEMS).filter(
        (item) => item.getAttribute("data-key") === navKey
      )[0];

      if (tmpItem && navValues.items) {
        Object.entries(navValues.items).forEach(([subnavKeyn, subnavItem]) => {
          const tmpSubnavItem = document.createElement("a");

          tmpSubnavItem.setAttribute("class", "header-main-item-subcategory");
          tmpSubnavItem.href = subnavItem.url;
          tmpSubnavItem.textContent = subnavItem.label;

          tmpItem.append(tmpSubnavItem);
        });
      }
    });
  });
}

function reset() {
  removeClass(HEADER_MAIN, CLASS_MOBILE_SUBNAV_IS_OPEN);
  removeClass(HEADER_MAIN, CLASS_MOBILE_SEARCH_IS_OPEN);
  isSearchOpen = false;
}

export default async function initNavHeader() {
  if (window.MqState && window.MqState.getState() > 2) {
    await getSubnav();
  } else {
    on(BTN_MOBILE_MENU, "click", getSubnav);
  }

  headerSlider();
  toggleMobileMenu();
  toggleMobileSubnavMenu();
  toggleSearchMobilePanel();

  on(OVERLAY_ELEMENT, "click", reset);

  eventEmitter.on(EventsTypes.MQ_STATE, (mqState) => {
    if (mqState > 2) {
      reset();
      hideOverlay();
    }
  });
}
