import { hasClass, addClass, removeClass } from "../../dom/classes";
import { on, off } from "../../dom/events";
import {
  ADDED_AD_CLASS,
  FILLED_AD_CLASS,
} from "../../constant/dom";
import { requestAnimationFrame } from "../../tools/requestAnimationFrame";
import eventEmitter, { EventsTypes } from "../../services/events/eventEmitter";
import { isJsonString } from "../../tools/string";

let createdAnchorAdsIndex = 0;
const JAD_CONFIG = window._GLOBALS?.jad_config;

export let ADSERVER_LOADED = false;

// LINKED to symfony manager bundle for the DOM logic
// LINKED to our UI bundle for the CSS
// HERE WE HAVE A DESIGN PATERN ISSUE REGARDING THE REFERENCES...
// Baby step ...
export async function createAnchorAds(referenceElement, adConfig = {}) {
  return new Promise((resolve) => {
    // create wrapper
    const adWrapper = document.createElement("div");
    adWrapper.className = "ad-placement";

    if (adConfig.isSticky) {
      adWrapper.className += " ad-sticky";
    }

    if (adConfig.hasPlaceholder) {
      adWrapper.className += " ad-placeholder";
    }

    if (adConfig.positionName) {
      adWrapper.className += ` ad-placement-${adConfig.positionName}`;
    }

    // create logo
    const adLogo = document.createElement("img");
    adLogo.className = "ad-logo";
    adLogo.src = JAD_CONFIG?.ad_logo_src;

    // create container
    const adContainer = document.createElement("div");
    adContainer.className = "ad-container";

    // create title
    let adTitle = null;
    if (JAD_CONFIG?.ad_title) {
      adTitle = document.createElement("div");
      adTitle.className = "ad-title";
      adTitle.textContent = JAD_CONFIG?.ad_title;
    }

    // create anchor
    const adAnchor = document.createElement("div");
    adAnchor.id = `${adConfig.positionName}-${createdAnchorAdsIndex++}`;
    adAnchor.className = `ad-item ${ADDED_AD_CLASS}`;
    adAnchor.dataset.position = adConfig.position;

    // append
    if (adTitle) {
      adContainer.appendChild(adTitle);
    }
    adContainer.appendChild(adAnchor);
    adWrapper.appendChild(adLogo);
    adWrapper.appendChild(adContainer);

    // inject
    requestAnimationFrame(() => {
      referenceElement.after(adWrapper);
      resolve(adWrapper);
    });
  });
}

export function triggerAdserverLoaded() {
  eventEmitter.emit(EventsTypes.ADSERVER_LOADED);
  ADSERVER_LOADED = true;
}

export function runAfterAdserverLoad() {
  let TIMEOUT = null;
  return new Promise((resolve, reject) => {
    if (!ADSERVER_LOADED) {
      TIMEOUT = setTimeout(() => {
        console.error('WARNING : ADSERVER_LOADED event was never sent. Use triggerAdserverLoaded function or trigger ADSERVER_LOADED custom event when adserver is loaded');
        reject();
      }, 10000);
    }

    if (ADSERVER_LOADED) {
      if (TIMEOUT) {
        clearTimeout(TIMEOUT);
        TIMEOUT = null;
      }
      resolve();
    } else {
      eventEmitter.once(EventsTypes.ADSERVER_LOADED, () => {
        if (TIMEOUT) {
          clearTimeout(TIMEOUT);
          TIMEOUT = null;
        }
        ADSERVER_LOADED = true;
        resolve();
      });
    }
  })
}

export async function handleAdsArticleInjection(paragraph) {
  return new Promise((resolve) => {
    // remove the last item to avoid getting an ads after the content (Outbrain compliancy)
    const paragraphOk = paragraph.slice(0, -2);

    const MODULO = JAD_CONFIG?.article_inread_added_interval || 3;

    if (paragraphOk.length < MODULO) {
      resolve();
      return;
    }

    const promiseIRA = [];

    paragraphOk.forEach(async (item, index) => {
      const number = index + 1;

      if (number % MODULO === 0) {
        switch (number / MODULO) {
          case 1:
            promiseIRA.push(new Promise((resolve) => {
              createAnchorAds(item, {
                position: "inread1",
                positionName: "inread1",
                isSticky: true,
                hasPlaceholder: true,
              }).then(() => {
                resolve();
              });
            }))
            break;
          case 2:
            promiseIRA.push(new Promise((resolve) => {
              createAnchorAds(item, {
                position: "inread2",
                positionName: "inread2",
                isSticky: true,
                hasPlaceholder: true,
              }).then(() => {
                resolve();
              });
            }));
            break;
          default:
            promiseIRA.push(new Promise((resolve) => {
              createAnchorAds(item, {
                position: "inread_added",
                positionName: "inread-added",
                isSticky: true,
                hasPlaceholder: true,
              }).then(() => {
                resolve();
              });
            }));
        }
      }
    });

    Promise.all(promiseIRA).then(() => {
      resolve();
    });
  })
}

export function handleStickyHeaderAd() {
  if (window.MqState?.getState() > 1) {
    return false;
  }

  const headerAdContainer = document.querySelector(".ad-placement-header");

  if (!headerAdContainer) {
    return false;
  }

  let sticked = false;
  const HEADER_AD_WRAPPER = document.querySelector(".ad-placement-header");
  const HEADER_AD_WRAPPER_OFFSET_TOP = HEADER_AD_WRAPPER.getBoundingClientRect().top;
  const MAIN_PLAYER = document.getElementById("player-main");
  const HEADER_AD_CONTAINER = HEADER_AD_WRAPPER.querySelector(".ad-container");
  const HEADER_AD_CONTAINER_HEIGHT = MAIN_PLAYER
    ? `${
        MAIN_PLAYER.getBoundingClientRect().top -
        HEADER_AD_WRAPPER_OFFSET_TOP -
        20
      }px`
    : null;

  on(window, "scroll", function stickHeaderAd() {
    if (
      hasClass(HEADER_AD_WRAPPER, FILLED_AD_CLASS) &&
      window.scrollY >= HEADER_AD_WRAPPER_OFFSET_TOP &&
      !sticked
    ) {
      addClass(HEADER_AD_WRAPPER, "ad-sticky-header");

      if (HEADER_AD_CONTAINER && HEADER_AD_CONTAINER_HEIGHT) {
        HEADER_AD_CONTAINER.style.height = HEADER_AD_CONTAINER_HEIGHT;
      }

      sticked = true;
      off(window, "scroll", stickHeaderAd);
    }
  });
}

////////////////////////////// HANDLE VIDEO ADS PARTS //////////////////////////////////////
async function getVideoAds(
  adPosition,
  playerPosition,
  anchorId,
  targetingFromVideoData
) {
  let playerPositionName = playerPosition;

  if (!playerPosition) {
    playerPositionName = "playercontent";
  }

  await runAfterAdserverLoad();

  return new Promise((resolve) => {
    // handle video targeting

    const videoAdsName = `${adPosition}_${playerPositionName}`;

    const positionAdskey = `${videoAdsName}/${anchorId}`;

    const targeting = isJsonString(targetingFromVideoData) ? JSON.parse(targetingFromVideoData) : null;

    window.jad.cmd.push(function () {
      if (targeting) {
        window.jad.public.setTargetingOnPosition(videoAdsName, {
          ...targeting,
        });
      }

      window.jad.public.getDailymotionAdsParamsForScript(
        [`${positionAdskey}`],
        (adsParams) => {
          resolve(adsParams[`${positionAdskey}`]);
        }
      );
    });
  });
}

export async function getPrerollConfiguration(data) {
  let tag = false;

  if (JAD_CONFIG?.use_preroll_fallback) {
    return { tag: getDefaultVastUrl() }
  }

  try {
    tag = await getVideoAds(
      "preroll",
      data.position,
      data.anchorId,
      data.keywords
    );
  } catch (e) {
    // eslint-disable-next-line no-console
    console.warn("getPrerollConfiguration error : ", e);
  }

  return { tag };
}

const getDefaultVastUrl = () => {
  if (!JAD_CONFIG?.keywords || !JAD_CONFIG?.page) {
    return false;
  }

  const cust_params = ['fallback=true'];

  for (const [key, value] of Object.entries(JAD_CONFIG.keywords)) {
    cust_params.push(`${key}=${value.join(',')}`);
  }

  const dfpAdunitTag = `${JAD_CONFIG.page}`;

  // the patern is : /ADUNIT_ID/ADUNIT_TOP_LEVEL/SUBLEVEL (encodé) + / (pas encodé) + key1=value&key2=value1,value2
  return `${encodeURIComponent(dfpAdunitTag)}/${encodeURIComponent(cust_params.join('&'))}`;
};
