import { hit } from "../tracking"
import { hitPixel } from "../tracking/pixel"
import { on } from "../../dom/events";
import handInterSectionObserver from "../../tools/intersectionObserverItems";
import { runAfterDeobfuscation } from "../../services/seo/deobfuscation";

function setPixelUrl(pixel, action) {
  return `${pixel}${action}&domain=${window.location.host}&referrer=${document.referrer}`
}

function handleNativeAnchorElementEvent() {
  const elementSet = new Set();

  const elementsToTrack = window.nativeAnchorTrackingData?.elements;

  if (!elementsToTrack || !Object.keys(elementsToTrack).length) {
    return false;
  }

  Object.keys(elementsToTrack).forEach(item => {
    // We do that to avoid sending double hit in AJAX mode when DOM is retrived for the seconde time or more
    if (elementSet.has(item)) {
      elementSet.delete(item);
    } else {
      elementSet.add(item);
    }
  });

  elementSet.forEach(item => {
    /*
    EVENT ACTION (eventAction in JAN data-set can be either a single string or an array of string values):
      - load : send a hit when the element is present in the document
      - view : send a hit when the element is visible by the user (in the viewport)
      - click : send a hit when the element is clicked by the user
    */

    const tackingObj = elementsToTrack[item];

    if (tackingObj) {
      const ELEMENT = document.getElementById(`${item}`);

      if (!ELEMENT) {
        console.warn(`${item} doesn't exist in DOM`);
        return false;
      }

      const GA_OBJ = {...tackingObj?.ga};
      const PIXEL_BASE_URL = tackingObj?.pixel;

      const GA_EVENT_ACTION = tackingObj?.meta?.action;
      const GA_EVENT_CATEGORY = tackingObj?.meta?.category;

      delete GA_OBJ.eventAction;
      delete GA_OBJ.eventName;

      // Here, impression is used as "element is present in the page on load".
      if (GA_EVENT_ACTION.includes('load') || GA_EVENT_ACTION === "load") {
        hit(`${GA_EVENT_CATEGORY}_load`, GA_OBJ);
        hitPixel(setPixelUrl(PIXEL_BASE_URL, 'load'));
      }

      // Here it's used as a "impression on view" meaning that the element is in the viewport.
      // We use intersectionObserver to trigger the hit.
      // WARNING => EventAction === "impression" will be deprecated soon and will be removed.
      if (GA_EVENT_ACTION.includes('view') || GA_EVENT_ACTION === "view" || GA_EVENT_ACTION === "impression") {
        handInterSectionObserver([{
          element: ELEMENT,
          callback: (entry, observer) => {
            if (entry.isIntersecting) {
              hit(`${GA_EVENT_CATEGORY}_view`, GA_OBJ);
              hitPixel(setPixelUrl(PIXEL_BASE_URL, 'view'));
              observer.unobserve(ELEMENT);
            }
          },
          options: {
            threshold: 0.5
          }
        }])
      }

      if (GA_EVENT_ACTION.includes('click') || GA_EVENT_ACTION === "click") {
        on(ELEMENT, 'click', () => {
          hit(`${GA_EVENT_CATEGORY}_click`, GA_OBJ);
          hitPixel(setPixelUrl(PIXEL_BASE_URL, 'click'));
        });
      }
    }
  });
}

export default function trackNativeAnchorElementEvent() {
  runAfterDeobfuscation(() => {
    handleNativeAnchorElementEvent();
  })
}
