/**
 * Roller Layout Section Stories
 *
 * Horizontal scrolling carousel/slider section layout.
 * 
 * This section extends base_layout.html.twig and embeds the roller module
 * to create a horizontally scrolling content carousel. Commonly used for
 * video playlists, image galleries, and content recommendations.
 * 
 * NOTE: Sections are more complex than modules and typically require
 * customization via {% block %} overrides in consuming projects.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters, TitleMarkup } from '../../../.storybook/types';

interface RollerEntity {
  id: number;
  title: string;
  description: string;
  thumbnail: string;
}

interface LayoutRollerArgs {
  title_text: string;
  title_markup: TitleMarkup;
  has_cta: boolean;
  cta_text?: string;
  cta_href?: string;
  offset: number;
  no_nav: boolean;
  item_count: number;
  entities: RollerEntity[];
}

const meta: Meta<LayoutRollerArgs> = {
  title: 'Sections/Dynamic Content/Roller Layout',
  parameters: {
    docs: {
      description: {
        component: `Horizontal scrolling carousel section. Extends base_layout and embeds roller module for smooth horizontal content navigation.
        
**Usage in projects:**
\`\`\`twig
{% extends '@Ui/section/dynamic_content/layout_roller.html.twig' %}

{% set block_path = 'your_blocks.html.twig' %}

{# block_path template should define 'item' block #}
\`\`\`

The roller supports:
- Left/right navigation buttons
- Configurable scroll offset
- RTL support
- Optional navigation hiding`,
      },
    },
    twig: {
      template: '_stories/section-roller-story.html.twig',
      componentPath: 'section/dynamic_content/layout_roller',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    title_text: {
      control: 'text',
      description: 'Section title text',
      table: { category: 'Content' },
    },
    title_markup: {
      control: 'select',
      options: ['h1', 'h2', 'h3', 'div'],
      description: 'HTML tag for section title',
      table: { category: 'Semantic' },
    },
    has_cta: {
      control: 'boolean',
      description: 'Show CTA button at bottom',
      table: { category: 'CTA' },
    },
    cta_text: {
      control: 'text',
      description: 'CTA button text',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    cta_href: {
      control: 'text',
      description: 'CTA button URL',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    offset: {
      control: 'number',
      description: 'Scroll offset in pixels for navigation',
      table: { category: 'Roller Config' },
    },
    no_nav: {
      control: 'boolean',
      description: 'Hide navigation buttons',
      table: { category: 'Roller Config' },
    },
    item_count: {
      control: 'number',
      description: 'Number of demo items to display',
      table: { category: 'Demo' },
    },
  },
};

export default meta;
type Story = StoryObj<LayoutRollerArgs>;

// Helper to generate mock entities
const createMockEntities = (count: number): RollerEntity[] => {
  return Array.from({ length: count }, (_, i) => ({
    id: i + 1,
    title: `Slide ${i + 1}`,
    description: `Content for slide ${i + 1}`,
    thumbnail: `https://picsum.photos/300/200?random=${i + 1}`,
  }));
};

export const Default: Story = {
  args: {
    title_text: 'Roller Section',
    title_markup: 'h2',
    has_cta: false,
    offset: 15,
    no_nav: false,
    item_count: 8,
    entities: createMockEntities(8),
  },
};

export const ManyItems: Story = {
  args: {
    title_text: 'Video Playlist',
    title_markup: 'h2',
    has_cta: false,
    offset: 15,
    no_nav: false,
    item_count: 15,
    entities: createMockEntities(15),
  },
};

export const WithCTA: Story = {
  args: {
    title_text: 'Featured Content',
    title_markup: 'h2',
    has_cta: true,
    cta_text: 'View All',
    cta_href: '/all-content',
    offset: 15,
    no_nav: false,
    item_count: 10,
    entities: createMockEntities(10),
  },
};

export const WithoutNavigation: Story = {
  args: {
    title_text: 'Auto-scroll Carousel',
    title_markup: 'h2',
    has_cta: false,
    offset: 15,
    no_nav: true,
    item_count: 8,
    entities: createMockEntities(8),
  },
};

export const LargeOffset: Story = {
  args: {
    title_text: 'Large Scroll Steps',
    title_markup: 'h2',
    has_cta: false,
    offset: 30,
    no_nav: false,
    item_count: 12,
    entities: createMockEntities(12),
  },
};

export const FewItems: Story = {
  args: {
    title_text: 'Small Collection',
    title_markup: 'h2',
    has_cta: false,
    offset: 15,
    no_nav: false,
    item_count: 4,
    entities: createMockEntities(4),
  },
};
