/**
 * Grid Layout Section Stories
 *
 * Grid-based section layout for displaying content in responsive columns.
 * 
 * This section extends base_layout.html.twig and displays items in a grid
 * with configurable columns and gap sizes. Commonly used for article lists,
 * product grids, and gallery layouts.
 * 
 * NOTE: Sections are more complex than modules and typically require
 * customization via {% block %} overrides in consuming projects.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters, TitleMarkup } from '../../../.storybook/types';

interface GridEntity {
  id: number;
  title: string;
  description: string;
}

interface LayoutGridSimpleArgs {
  title_text: string;
  title_markup: TitleMarkup;
  gd_l_number: '2' | '3' | '4' | '5' | '6';
  gap_size: '10' | '15' | '20' | '30' | '40';
  has_cta: boolean;
  cta_text?: string;
  cta_href?: string;
  item_count: number;
  entities: GridEntity[];
}

const meta: Meta<LayoutGridSimpleArgs> = {
  title: 'Sections/Dynamic Content/Grid Layout',
  parameters: {
    docs: {
      description: {
        component: `Grid-based section layout with responsive columns. Extends base_layout to provide title, content grid, and optional CTA button.
        
**Usage in projects:**
\`\`\`twig
{% extends '@Ui/section/dynamic_content/layout_grid_simple.html.twig' %}

{% block item %}
  {# Your item template here #}
  <article>{{ item.title }}</article>
{% endblock %}
\`\`\``,
      },
    },
    twig: {
      template: '_stories/section-grid-story.html.twig',
      componentPath: 'section/dynamic_content/layout_grid_simple',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    title_text: {
      control: 'text',
      description: 'Section title text',
      table: { category: 'Content' },
    },
    title_markup: {
      control: 'select',
      options: ['h1', 'h2', 'h3', 'div'],
      description: 'HTML tag for section title',
      table: { category: 'Semantic' },
    },
    gd_l_number: {
      control: 'select',
      options: ['2', '3', '4', '5', '6'],
      description: 'Number of columns on large screens',
      table: { category: 'Layout' },
    },
    gap_size: {
      control: 'select',
      options: ['10', '15', '20', '30', '40'],
      description: 'Gap size between grid items (px)',
      table: { category: 'Layout' },
    },
    has_cta: {
      control: 'boolean',
      description: 'Show CTA button at bottom',
      table: { category: 'CTA' },
    },
    cta_text: {
      control: 'text',
      description: 'CTA button text',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    cta_href: {
      control: 'text',
      description: 'CTA button URL',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    item_count: {
      control: 'number',
      description: 'Number of demo items to display',
      table: { category: 'Demo' },
    },
  },
};

export default meta;
type Story = StoryObj<LayoutGridSimpleArgs>;

// Helper to generate mock entities
const createMockEntities = (count: number): GridEntity[] => {
  return Array.from({ length: count }, (_, i) => ({
    id: i + 1,
    title: `Item ${i + 1}`,
    description: `Description for item ${i + 1}`,
  }));
};

export const FourColumns: Story = {
  args: {
    title_text: 'Grid Section - 4 Columns',
    title_markup: 'h2',
    gd_l_number: '4',
    gap_size: '20',
    has_cta: false,
    item_count: 8,
    entities: createMockEntities(8),
  },
};

export const ThreeColumns: Story = {
  args: {
    title_text: 'Grid Section - 3 Columns',
    title_markup: 'h2',
    gd_l_number: '3',
    gap_size: '20',
    has_cta: false,
    item_count: 9,
    entities: createMockEntities(9),
  },
};

export const TwoColumns: Story = {
  args: {
    title_text: 'Grid Section - 2 Columns',
    title_markup: 'h2',
    gd_l_number: '2',
    gap_size: '30',
    has_cta: false,
    item_count: 6,
    entities: createMockEntities(6),
  },
};

export const WithCTA: Story = {
  args: {
    title_text: 'Featured Articles',
    title_markup: 'h2',
    gd_l_number: '4',
    gap_size: '20',
    has_cta: true,
    cta_text: 'View All Articles',
    cta_href: '/articles',
    item_count: 8,
    entities: createMockEntities(8),
  },
};

export const LargeGap: Story = {
  args: {
    title_text: 'Grid with Large Gaps',
    title_markup: 'h2',
    gd_l_number: '3',
    gap_size: '40',
    has_cta: false,
    item_count: 6,
    entities: createMockEntities(6),
  },
};

export const ManyItems: Story = {
  args: {
    title_text: 'Large Grid Section',
    title_markup: 'h2',
    gd_l_number: '4',
    gap_size: '20',
    has_cta: true,
    cta_text: 'Load More',
    cta_href: '/more',
    item_count: 16,
    entities: createMockEntities(16),
  },
};
