/**
 * Base Layout Section Stories
 *
 * Foundation template for all section layouts in UI-Bundle.
 * 
 * This is the abstract base template that all other section layouts extend from.
 * It provides the fundamental structure: container, title, content area, and CTA.
 * 
 * NOTE: This template is designed to be extended, not used directly in projects.
 * Use concrete implementations like layout_roller, layout_grid_simple, etc.
 * 
 * Available blocks for customization:
 * - section_title: Override title rendering (includes @Ui/component/title.html.twig by default)
 * - section_content: Main content area (empty by default, must be overridden)
 * - section_cta: Call-to-action button area (conditional rendering based on has_cta)
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { ButtonSize, StoryParameters, TitleMarkup } from '../../../.storybook/types';

interface BaseLayoutArgs {
  title_text: string;
  title_markup: TitleMarkup;
  is_big_section: boolean;
  has_cta: boolean;
  cta_text?: string;
  cta_href?: string;
  cta_type?: string;
  cta_size?: ButtonSize;
  section_classes: string;
}

const meta: Meta<BaseLayoutArgs> = {
  title: 'Sections/Dynamic Content/Base Layout',
  parameters: {
    docs: {
      description: {
        component: `Base template for all section layouts. Provides the foundational structure with title, content, and CTA patterns.
        
**Architecture:**
- Renders as \`<section>\` or \`<div>\` based on \`is_big_section\` variable
- Includes \`.mg-container\` class by default
- Integrates @Ui/component/title.html.twig for section titles
- Supports optional CTA button via @Ui/component/button.html.twig

**Usage in projects:**
\`\`\`twig
{% extends '@Ui/section/dynamic_content/base_layout.html.twig' %}

{% set title_text = 'My Section' %}
{% set title_markup = 'h2' %}
{% set has_cta = true %}
{% set cta_text = 'View More' %}
{% set cta_href = '/more' %}

{% block section_content %}
  {# Your custom content here #}
  <div class="mg-content">
    <!-- Content markup -->
  </div>
{% endblock %}
\`\`\`

**Available Blocks:**
- \`section_title\`: Override title rendering
- \`section_content\`: Main content area (required)
- \`section_cta\`: Override CTA rendering

**Variables:**
- \`is_big_section\`: boolean - Use <div> instead of <section> (default: false)
- \`section_classes\`: array - Additional CSS classes
- \`title_text\`, \`title_markup\`: Title configuration
- \`has_cta\`, \`cta_text\`, \`cta_href\`, \`cta_type\`, \`cta_size\`: CTA configuration`,
      },
    },
    twig: {
      template: '_stories/dynamic_content-base_layout-story.html.twig',
      componentPath: 'section/dynamic_content/base_layout',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    title_text: {
      control: 'text',
      description: 'Section title text',
      table: { category: 'Content' },
    },
    title_markup: {
      control: 'select',
      options: ['h1', 'h2', 'h3', 'div'],
      description: 'HTML tag for section title',
      table: { category: 'Semantic' },
    },
    is_big_section: {
      control: 'boolean',
      description: 'Use <div> instead of <section> tag',
      table: { category: 'Semantic' },
    },
    has_cta: {
      control: 'boolean',
      description: 'Show CTA button at bottom',
      table: { category: 'CTA' },
    },
    cta_text: {
      control: 'text',
      description: 'CTA button text',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    cta_href: {
      control: 'text',
      description: 'CTA button URL',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    cta_type: {
      control: 'select',
      options: ['primary', 'secondary', 'tertiary'],
      description: 'CTA button type/style',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    cta_size: {
      control: 'select',
      options: ['small', 'medium', 'large'],
      description: 'CTA button size',
      table: { category: 'CTA' },
      if: { arg: 'has_cta', truthy: true },
    },
    section_classes: {
      control: 'text',
      description: 'Additional CSS classes (space-separated)',
      table: { category: 'Styling' },
    },
  },
};

export default meta;
type Story = StoryObj<BaseLayoutArgs>;

export const Default: Story = {
  args: {
    title_text: 'Base Layout Section',
    title_markup: 'h2',
    is_big_section: false,
    has_cta: false,
    section_classes: '',
  },
};

export const WithCTA: Story = {
  args: {
    title_text: 'Section with Call-to-Action',
    title_markup: 'h2',
    is_big_section: false,
    has_cta: true,
    cta_text: 'View All',
    cta_href: '/view-all',
    cta_type: 'primary',
    cta_size: 'medium',
    section_classes: '',
  },
};

export const AsDivElement: Story = {
  args: {
    title_text: 'Rendered as Div Element',
    title_markup: 'h2',
    is_big_section: true,
    has_cta: false,
    section_classes: '',
  },
};

export const WithCustomClasses: Story = {
  args: {
    title_text: 'Section with Custom Classes',
    title_markup: 'h2',
    is_big_section: false,
    has_cta: false,
    section_classes: 'custom-theme dark-mode',
  },
};

export const H1Title: Story = {
  args: {
    title_text: 'Page-Level Section Title',
    title_markup: 'h1',
    is_big_section: false,
    has_cta: false,
    section_classes: '',
  },
};

export const CompletExample: Story = {
  args: {
    title_text: 'Complete Section Example',
    title_markup: 'h2',
    is_big_section: false,
    has_cta: true,
    cta_text: 'Explore More',
    cta_href: '/explore',
    cta_type: 'secondary',
    cta_size: 'large',
    section_classes: 'featured-section',
  },
};
