/**
 * Video Card Module Stories
 *
 * Video card component for displaying video thumbnails with metadata.
 * 
 * This module displays a video thumbnail with play icon, title, and optional labels.
 * It supports both standard 16:9 and short-form 9:16 aspect ratios.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface VideoCardArgs {
  video_title: string;
  video_link?: string;
  video_thumb: string;
  video_play_picto: boolean;
  is_short: boolean;
  current_video: boolean;
  card_title_markup: 'div' | 'h2' | 'h3' | 'h4' | 'p';
  video_id?: string;
  card_classes: string[];
}

const meta: Meta<VideoCardArgs> = {
  title: 'Modules/Video Card',
  parameters: {
    docs: {
      description: {
        component: 'Video card displaying thumbnail, title, and optional play icon. Supports standard and short-form video formats.',
      },
    },
    twig: {
      template: 'module/video/video_card.html.twig',
      renderMode: 'include',
      componentPath: 'module/video/video_card',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    video_title: {
      control: 'text',
      description: 'Video title text',
      table: { category: 'Content' },
    },
    video_link: {
      control: 'text',
      description: 'URL for the video page',
      table: { category: 'Navigation' },
    },
    video_thumb: {
      control: 'text',
      description: 'Video thumbnail image URL',
      table: { category: 'Media' },
    },
    video_play_picto: {
      control: 'boolean',
      description: 'Show play icon on thumbnail',
      table: { category: 'Visual' },
    },
    is_short: {
      control: 'boolean',
      description: 'Short-form video format (9:16 ratio)',
      table: { category: 'Visual' },
    },
    current_video: {
      control: 'boolean',
      description: 'Mark as currently playing video',
      table: { category: 'State' },
    },
    card_title_markup: {
      control: 'select',
      options: ['div', 'h2', 'h3', 'h4', 'p'],
      description: 'HTML tag for title element',
      table: { category: 'Semantic' },
    },
    video_id: {
      control: 'text',
      description: 'Video identifier for data attribute',
      table: { category: 'Data' },
    },
    card_classes: {
      control: 'object',
      description: 'Additional CSS classes',
      table: { category: 'Style' },
    },
  },
};

export default meta;
type Story = StoryObj<VideoCardArgs>;

export const Default: Story = {
  args: {
    video_title: 'Amazing Gaming Moments Compilation',
    video_link: '/videos/amazing-gaming-moments',
    video_thumb: 'https://picsum.photos/640/360',
    video_play_picto: true,
    is_short: false,
    current_video: false,
    card_title_markup: 'div',
    video_id: 'vid-12345',
    card_classes: [],
  },
};

export const CurrentVideo: Story = {
  args: {
    ...Default.args,
    video_title: 'Currently Playing: Epic Boss Battle',
    current_video: true,
  },
};

export const ShortFormVideo: Story = {
  args: {
    ...Default.args,
    video_title: 'Quick Gaming Tip #42',
    is_short: true,
    video_thumb: 'https://picsum.photos/360/640',
  },
};

export const WithoutPlayIcon: Story = {
  args: {
    ...Default.args,
    video_title: 'Video Without Play Button',
    video_play_picto: false,
  },
};

export const LongTitle: Story = {
  args: {
    ...Default.args,
    video_title: 'This is a Very Long Video Title That Might Need to Wrap to Multiple Lines in the Card Layout',
  },
};

export const WithH3Title: Story = {
  args: {
    ...Default.args,
    video_title: 'Semantic H3 Video Title',
    card_title_markup: 'h3',
  },
};

export const WithCustomClasses: Story = {
  args: {
    ...Default.args,
    video_title: 'Featured Video',
    card_classes: ['featured', 'highlighted'],
  },
};

export const WithoutLink: Story = {
  args: {
    video_title: 'Video Title Without Link',
    video_thumb: 'https://picsum.photos/640/360',
    video_play_picto: true,
    is_short: false,
    current_video: false,
    card_title_markup: 'div',
    card_classes: [],
  },
};
