/**
 * Search Module Stories
 *
 * Search form with input field and submit button.
 * 
 * A simple, accessible search form component with icon, text input, and submit button.
 * Includes JavaScript integration hooks for live search functionality.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface SearchArgs {
  search_path: string;
  place_holder: string;
  search_method: 'get' | 'post';
  search_input_name: string;
  search_input_id: string;
}

const meta: Meta<SearchArgs> = {
  title: 'Modules/Search',
  parameters: {
    docs: {
      description: {
        component: `Accessible search form with icon, input, and submit button.
        
**Features:**
- Search icon label
- Text input with placeholder
- Submit button (disabled by default until input)
- Configurable form method (GET/POST)
- JavaScript hooks for search enhancements
- Accessibility labels

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/search/search.html.twig' with {
    search_path: path('search'),
    place_holder: 'Search...',
    search_method: 'get'
} only %}
\`\`\``,
      },
    },
    twig: {
      template: 'module/search/search.html.twig',
      renderMode: 'include',
      componentPath: 'module/search/search',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    search_path: {
      control: 'text',
      description: 'Form action URL',
      table: { category: 'Form' },
    },
    place_holder: {
      control: 'text',
      description: 'Input placeholder text',
      table: { category: 'Content' },
    },
    search_method: {
      control: 'select',
      options: ['get', 'post'],
      description: 'Form HTTP method',
      table: { category: 'Form' },
    },
    search_input_name: {
      control: 'text',
      description: 'Input field name attribute',
      table: { category: 'Form' },
    },
    search_input_id: {
      control: 'text',
      description: 'Input field ID attribute',
      table: { category: 'Form' },
    },
  },
};

export default meta;
type Story = StoryObj<SearchArgs>;

export const Default: Story = {
  args: {
    search_path: '/search',
    place_holder: 'Search the website...',
    search_method: 'post',
    search_input_name: 'q',
    search_input_id: 'search_bar',
  },
};

export const GetMethod: Story = {
  args: {
    search_path: '/search',
    place_holder: 'Search...',
    search_method: 'get',
    search_input_name: 'q',
    search_input_id: 'search_bar',
  },
};

export const ShortPlaceholder: Story = {
  args: {
    search_path: '/search',
    place_holder: 'Search',
    search_method: 'post',
    search_input_name: 'q',
    search_input_id: 'search_bar',
  },
};

export const LongPlaceholder: Story = {
  args: {
    search_path: '/search',
    place_holder: 'Search for games, reviews, news, and more...',
    search_method: 'post',
    search_input_name: 'q',
    search_input_id: 'search_bar',
  },
};

export const CustomInputName: Story = {
  args: {
    search_path: '/search',
    place_holder: 'Search...',
    search_method: 'post',
    search_input_name: 'query',
    search_input_id: 'site_search',
  },
};
