/**
 * Roller Module Stories
 *
 * Horizontal scrolling carousel with navigation buttons.
 * 
 * Core roller component providing smooth horizontal scrolling with left/right
 * navigation buttons. Content is provided via block override. Used by many
 * other modules (link_bar, subnav, etc.) as the scrolling foundation.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface RollerItem {
  id: number;
  title: string;
  color: string;
}

interface RollerArgs {
  items: RollerItem[];
  offset: number;
  no_nav: boolean;
  icon_left: string;
  icon_right: string;
  classes: string[];
  website_direction_rtl: boolean;
}

const meta: Meta<RollerArgs> = {
  title: 'Modules/Roller',
  parameters: {
    docs: {
      description: {
        component: `Horizontal scrolling carousel component with navigation controls.
        
**Features:**
- Left/right navigation buttons
- Smooth scrolling behavior
- Configurable scroll offset
- RTL language support
- Optional navigation hiding
- Custom icons for navigation
- Content via block override

**Usage in projects:**
\`\`\`twig
{% embed '@Ui/module/roller/roller.html.twig' with {
    offset: 15,
    no_nav: false
} %}
    {% block slider %}
        <!-- Your scrollable content here -->
    {% endblock %}
{% endembed %}
\`\`\``,
      },
    },
    twig: {
      template: '_stories/roller-story.html.twig',
      componentPath: 'module/roller/roller',
      argsToContext: (args) => args,
      jsModules: ['roller'],
    },
  } as StoryParameters,
  argTypes: {
    items: {
      control: 'object',
      description: 'Array of items to display',
      table: { category: 'Content' },
    },
    offset: {
      control: 'number',
      description: 'Scroll offset in pixels',
      table: { category: 'Navigation' },
    },
    no_nav: {
      control: 'boolean',
      description: 'Hide navigation buttons',
      table: { category: 'Navigation' },
    },
    icon_left: {
      control: 'text',
      description: 'Left navigation icon',
      table: { category: 'Visual' },
    },
    icon_right: {
      control: 'text',
      description: 'Right navigation icon',
      table: { category: 'Visual' },
    },
    classes: {
      control: 'object',
      description: 'Additional CSS classes',
      table: { category: 'Style' },
    },
    website_direction_rtl: {
      control: 'boolean',
      description: 'Right-to-left layout',
      table: { category: 'Localization' },
    },
  },
};

export default meta;
type Story = StoryObj<RollerArgs>;

// Mock data helper
const createRollerItems = (count = 8): RollerItem[] => {
  return Array.from({ length: count }, (_, i) => ({
    id: i + 1,
    title: `Item ${i + 1}`,
    color: ['#ef4444', '#3b82f6', '#10b981', '#f59e0b', '#8b5cf6', '#ec4899'][i % 6],
  }));
};

export const Default: Story = {
  args: {
    items: createRollerItems(8),
    offset: 15,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const ManyItems: Story = {
  args: {
    items: createRollerItems(20),
    offset: 15,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const NoNavigation: Story = {
  args: {
    items: createRollerItems(10),
    offset: 15,
    no_nav: true,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const LargeOffset: Story = {
  args: {
    items: createRollerItems(12),
    offset: 30,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const SmallOffset: Story = {
  args: {
    items: createRollerItems(12),
    offset: 5,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const RTLLayout: Story = {
  args: {
    items: createRollerItems(10),
    offset: 15,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: true,
  },
};

export const CustomIcons: Story = {
  args: {
    items: createRollerItems(8),
    offset: 15,
    no_nav: false,
    icon_left: 'arrow_left',
    icon_right: 'arrow_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const FewItems: Story = {
  args: {
    items: createRollerItems(4),
    offset: 15,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: [],
    website_direction_rtl: false,
  },
};

export const WithCustomClasses: Story = {
  args: {
    items: createRollerItems(8),
    offset: 15,
    no_nav: false,
    icon_left: 'chevron_left',
    icon_right: 'chevron_right',
    classes: ['custom-roller', 'featured'],
    website_direction_rtl: false,
  },
};
