/**
 * Player Card Module Stories
 *
 * Main player component that handles both internal (Dailymotion) and external (iframe) players.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface PlayerCardArgs {
  video_id: string;
  player_id: string;
  player_thumbnail: string;
  is_main_media: boolean;
  player_position: string;
}

const meta: Meta<PlayerCardArgs> = {
  title: 'Modules/Player/Player Card',
  parameters: {
    docs: {
      description: {
        component: `Main player container component that orchestrates internal (Dailymotion) and external (iframe) video players.

**Features:**
- Automatically switches between internal and external player based on video_id/playlist_id presence
- Supports Dailymotion internal player with configurable player_id
- Supports external iframe embeds for other video sources
- Displays thumbnail fallback when video is not yet available`,
      },
    },
    twig: {
      template: 'module/player/player_card.html.twig',
      renderMode: 'include',
      componentPath: 'module/player/player_card',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    video_id: {
      control: 'text',
      description: 'Dailymotion video ID',
      table: { category: 'Content' },
    },
    player_id: {
      control: 'text',
      description: 'Dailymotion player ID',
      table: { category: 'Player' },
    },
    player_thumbnail: {
      control: 'text',
      description: 'Thumbnail image URL',
      table: { category: 'Content' },
    },
    is_main_media: {
      control: 'boolean',
      description: 'Whether this is the main media player on the page',
      table: { category: 'Behavior' },
    },
    player_position: {
      control: 'select',
      options: ['playertop', 'playercontent'],
      description: 'Player position for analytics',
      table: { category: 'Analytics' },
    },
  },
};

export default meta;
type Story = StoryObj<PlayerCardArgs>;

export const Default: Story = {
  args: {
    video_id: 'x8j9k2l',
    player_id: 'xbcde',
    player_thumbnail: 'https://picsum.photos/580/326',
    is_main_media: false,
    player_position: 'playercontent',
  },
};

export const MainPlayer: Story = {
  args: {
    video_id: 'x8j9k2l',
    player_id: 'xabcd',
    player_thumbnail: 'https://picsum.photos/580/326',
    is_main_media: true,
    player_position: 'playertop',
  },
};
