/**
 * Pagination Bis Module Stories
 *
 * Alternate pagination component with customizable item styling.
 *
 * A flexible pagination component that displays page numbers with current/disabled
 * states. Supports custom filtering, link generation via blocks, and styling.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface PaginationBisArgs {
  pagination_library: (number | string)[];
  pagination_current: number;
  pagination_filter?: (number | string)[] | null;
  pagination_container_classes?: string[];
  pagination_item_classes?: string[];
}

// Component metadata
const meta: Meta<PaginationBisArgs> = {
  title: 'Modules/Pagination/Pagination Bis',
  parameters: {
    docs: {
      description: {
        component: `Flexible pagination component with block-based link generation.

**Features:**
- Current page highlighting
- Disabled/filtered page handling
- Block-based link href override
- Custom CSS classes
- SEO link parameter support
- Flexible markup structure

**Usage in projects:**
\`\`\`twig
{% embed '@Ui/module/pagination/pagination_bis.html.twig' with {
    pagination_library: [1, 2, 3, 4, 5],
    pagination_current: 3
} %}
    {% block link_href %}{{ path('page', {page: item}) }}{% endblock %}
{% endembed %}
\`\`\``,
      },
    },
    twig: {
      template: '_stories/pagination-bis-story.html.twig',
      componentPath: 'module/pagination/pagination_bis',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    pagination_library: {
      control: 'object',
      description: 'Array of page numbers/labels',
      table: { category: 'Content' },
    },
    pagination_current: {
      control: 'number',
      description: 'Current page number',
      table: { category: 'State' },
    },
    pagination_filter: {
      control: 'object',
      description: 'Array of pages to disable/filter',
      table: { category: 'State' },
    },
    pagination_container_classes: {
      control: 'object',
      description: 'Container CSS classes',
      table: { category: 'Style' },
    },
    pagination_item_classes: {
      control: 'object',
      description: 'Item CSS classes',
      table: { category: 'Style' },
    },
  },
};

export default meta;
type Story = StoryObj<PaginationBisArgs>;

// Stories
export const Default: Story = {
  args: {
    pagination_library: [1, 2, 3, 4, 5, 6, 7],
    pagination_current: 3,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const FirstPage: Story = {
  args: {
    pagination_library: [1, 2, 3, 4, 5, 6, 7],
    pagination_current: 1,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const LastPage: Story = {
  args: {
    pagination_library: [1, 2, 3, 4, 5, 6, 7],
    pagination_current: 7,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const ManyPages: Story = {
  args: {
    pagination_library: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12],
    pagination_current: 6,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const WithPrevNext: Story = {
  args: {
    pagination_library: ['‹', 1, 2, 3, 4, 5, '›'],
    pagination_current: 3,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const WithEllipsis: Story = {
  args: {
    pagination_library: [1, 2, 3, '...', 8, 9, 10],
    pagination_current: 2,
    pagination_filter: ['...'],
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const ComplexPagination: Story = {
  args: {
    pagination_library: ['‹‹', '‹', 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, '›', '››'],
    pagination_current: 5,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const WithCustomClasses: Story = {
  args: {
    pagination_library: [1, 2, 3, 4, 5],
    pagination_current: 2,
    pagination_filter: null,
    pagination_container_classes: ['custom-pagination'],
    pagination_item_classes: ['custom-item'],
  },
};

export const ThreePages: Story = {
  args: {
    pagination_library: [1, 2, 3],
    pagination_current: 2,
    pagination_filter: null,
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};

export const WithFilteredPages: Story = {
  args: {
    pagination_library: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
    pagination_current: 5,
    pagination_filter: [1, 2, 8, 9, 10],
    pagination_container_classes: [],
    pagination_item_classes: [],
  },
};
