/**
 * Pagination Module Stories
 *
 * Pagination navigation for multi-page content.
 * 
 * This module displays page numbers with previous/next controls.
 * It shows a batch of page numbers with the current page highlighted.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { PaginationData, StoryParameters } from '../../../.storybook/types';

// Component metadata
const meta: Meta<PaginationData> = {
  title: 'Modules/Pagination',
  parameters: {
    docs: {
      description: {
        component: 'Pagination navigation for multi-page content with previous/next controls.',
      },
    },
    twig: {
      template: 'module/pagination/pagination.html.twig',
      renderMode: 'include',
      componentPath: 'module/pagination/pagination',
      argsToContext: (args) => ({ pagination: args }),
    },
  } as StoryParameters,
  argTypes: {
    currentPage: {
      control: 'number',
      description: 'Current active page number',
      table: { category: 'State' },
    },
    getPageCount: {
      control: 'number',
      description: 'Total number of pages',
      table: { category: 'State' },
    },
  },
};

export default meta;
type Story = StoryObj<PaginationData>;

// Helper to generate pagination object
const createPaginationData = (currentPage: number, pageCount: number): PaginationData => {
  const pages = [];
  const startPage = Math.max(1, currentPage - 3);
  const endPage = Math.min(pageCount, currentPage + 3);
  
  for (let i = startPage; i <= endPage; i++) {
    pages.push({
      page: i,
      url: `/page/${i}`,
    });
  }
  
  return {
    getPageCount: pageCount,
    currentPage,
    pages,
    previousPage: currentPage > 1 ? { url: `/page/${currentPage - 1}`, page: currentPage - 1 } : null,
    nextPage: currentPage < pageCount ? { url: `/page/${currentPage + 1}`, page: currentPage + 1 } : null,
    firstPage: currentPage > 4 ? { url: '/page/1', page: 1 } : null,
    lastPage: currentPage < pageCount - 3 ? { url: `/page/${pageCount}`, page: pageCount } : null,
  };
};

// Stories
export const FirstPage: Story = {
  args: createPaginationData(1, 20),
};

export const MiddlePage: Story = {
  args: createPaginationData(10, 20),
};

export const LastPage: Story = {
  args: createPaginationData(20, 20),
};

export const FewPages: Story = {
  args: createPaginationData(2, 5),
};

export const SinglePageInMiddle: Story = {
  args: createPaginationData(5, 10),
};
