/**
 * Subnav Module Stories
 *
 * Sub-navigation roller component for secondary navigation menus.
 * 
 * This module creates a horizontally scrolling sub-navigation bar that embeds
 * custom content in a roller component. Commonly used below primary navigation
 * for category filters, subcategory links, or contextual navigation.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface SubnavItem {
  label: string;
  href: string;
  active: boolean;
}

interface SubnavArgs {
  items: SubnavItem[];
  no_nav: boolean;
  offset: number;
  dataAttr: Record<string, string>;
}

const meta: Meta<SubnavArgs> = {
  title: 'Modules/Navigation/Subnav',
  parameters: {
    docs: {
      description: {
        component: `Horizontal scrolling sub-navigation component.
        
**Features:**
- Embeds custom HTML content in roller
- Optional navigation arrows (disabled by default)
- Configurable scroll offset
- Flexible content via block override
- Custom data attributes support

**Usage in projects:**
\`\`\`twig
{% embed '@Ui/module/nav/subnav.html.twig' with {
    no_nav: true,
    offset: 15
} %}
    {% block subnav_items %}
        <a href="/cat1">Category 1</a>
        <a href="/cat2">Category 2</a>
    {% endblock %}
{% endembed %}
\`\`\`

**Note:** This module requires content to be provided via the \`subnav_items\` block.`,
      },
    },
    twig: {
      template: '_stories/subnav-story.html.twig',
      componentPath: 'module/nav/subnav',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    items: {
      control: 'object',
      description: 'Array of subnav items to display',
      table: { category: 'Content' },
    },
    no_nav: {
      control: 'boolean',
      description: 'Hide navigation arrows',
      table: { category: 'Navigation' },
    },
    offset: {
      control: 'number',
      description: 'Scroll offset in pixels',
      table: { category: 'Navigation' },
    },
    dataAttr: {
      control: 'object',
      description: 'Custom data attributes',
      table: { category: 'Data' },
    },
  },
};

export default meta;
type Story = StoryObj<SubnavArgs>;

// Mock data helpers
const createSubnavItems = (count = 6): SubnavItem[] => {
  const categories = [
    'All', 'Action', 'Adventure', 'RPG', 'Strategy', 
    'Sports', 'Racing', 'Simulation', 'Puzzle', 'Horror'
  ];
  return Array.from({ length: count }, (_, i) => ({
    label: categories[i] || `Filter ${i + 1}`,
    href: `/category/${i}`,
    active: i === 0,
  }));
};

export const Default: Story = {
  args: {
    items: createSubnavItems(6),
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};

export const WithNavigation: Story = {
  args: {
    items: createSubnavItems(10),
    no_nav: false,
    offset: 15,
    dataAttr: {},
  },
};

export const ManyItems: Story = {
  args: {
    items: createSubnavItems(15),
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};

export const FewItems: Story = {
  args: {
    items: createSubnavItems(3),
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};

export const LargeScrollOffset: Story = {
  args: {
    items: createSubnavItems(8),
    no_nav: false,
    offset: 30,
    dataAttr: {},
  },
};

export const SmallScrollOffset: Story = {
  args: {
    items: createSubnavItems(8),
    no_nav: false,
    offset: 5,
    dataAttr: {},
  },
};

export const WithActiveItem: Story = {
  args: {
    items: [
      { label: 'All Games', href: '/all', active: false },
      { label: 'Action', href: '/action', active: false },
      { label: 'RPG', href: '/rpg', active: true },
      { label: 'Strategy', href: '/strategy', active: false },
      { label: 'Sports', href: '/sports', active: false },
      { label: 'Indie', href: '/indie', active: false },
    ],
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};

export const CategoryFilters: Story = {
  args: {
    items: [
      { label: 'Latest', href: '/latest', active: true },
      { label: 'Most Popular', href: '/popular', active: false },
      { label: 'Editor\'s Choice', href: '/editors', active: false },
      { label: 'Community Picks', href: '/community', active: false },
      { label: 'Coming Soon', href: '/upcoming', active: false },
    ],
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};

export const PlatformFilters: Story = {
  args: {
    items: [
      { label: 'All Platforms', href: '/all', active: true },
      { label: 'PlayStation 5', href: '/ps5', active: false },
      { label: 'Xbox Series X|S', href: '/xbox', active: false },
      { label: 'Nintendo Switch', href: '/switch', active: false },
      { label: 'PC', href: '/pc', active: false },
      { label: 'Mobile', href: '/mobile', active: false },
    ],
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};

export const WithCustomDataAttributes: Story = {
  args: {
    items: createSubnavItems(6),
    no_nav: true,
    offset: 15,
    dataAttr: {
      'data-tracking': 'subnav-filter',
      'data-context': 'category-page',
    },
  },
};

export const LongLabels: Story = {
  args: {
    items: [
      { label: 'All Video Games', href: '/all', active: false },
      { label: 'Action & Adventure', href: '/action', active: false },
      { label: 'Role-Playing Games', href: '/rpg', active: true },
      { label: 'Strategy & Tactics', href: '/strategy', active: false },
      { label: 'Sports & Racing', href: '/sports', active: false },
    ],
    no_nav: true,
    offset: 15,
    dataAttr: {},
  },
};
