/**
 * Link Bar Module Stories
 *
 * Horizontal scrolling navigation bar embedded in a roller component.
 * 
 * This module creates a horizontally scrolling link bar commonly used for:
 * - Category navigation
 * - Tag filters
 * - Partner/sponsor logos
 * - Quick access links
 * 
 * The module automatically embeds items in a roller component for smooth scrolling.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface LinkBarTextItem {
  label: string;
  route_name?: string;
  route_params?: Record<string, unknown>;
  url?: string;
  seoLink?: string;
  tracking?: string;
  data_attr?: Record<string, string>;
  element_id?: string;
}

interface LinkBarImageItem {
  label: string;
  img: string;
  alt: string;
  width: number;
  height: number;
  url: string;
  seoLink: string;
  no_label?: boolean;
}

type LinkBarItem = LinkBarTextItem | LinkBarImageItem;

interface LinkBarArgs {
  link_bar: LinkBarItem[];
  link_bar_classes: string[];
  icon_left?: string;
  icon_right?: string;
}

const meta: Meta<LinkBarArgs> = {
  title: 'Modules/Navigation/Link Bar',
  parameters: {
    docs: {
      description: {
        component: `Horizontal scrolling link bar for navigation or content filtering.
        
**Features:**
- Horizontal scrolling via roller component
- Support for text-only or image+text links
- External and internal link support
- Optional tracking attributes
- Custom data attributes per item
- Responsive design (no navigation arrows by default)

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/nav/link_bar.html.twig' with {
    link_bar: [
        { label: 'News', route_name: 'news' },
        { label: 'Reviews', route_name: 'reviews' },
    ]
} only %}
\`\`\``,
      },
    },
    twig: {
      template: 'module/nav/link_bar.html.twig',
      renderMode: 'include',
      componentPath: 'module/nav/link_bar',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    link_bar: {
      control: 'object',
      description: 'Array of link items to display',
      table: { category: 'Content' },
    },
    link_bar_classes: {
      control: 'object',
      description: 'Additional CSS classes for container',
      table: { category: 'Style' },
    },
    icon_left: {
      control: 'text',
      description: 'Left navigation icon (if nav enabled)',
      table: { category: 'Navigation' },
    },
    icon_right: {
      control: 'text',
      description: 'Right navigation icon (if nav enabled)',
      table: { category: 'Navigation' },
    },
  },
};

export default meta;
type Story = StoryObj<LinkBarArgs>;

// Mock data helpers
const createTextLinks = (count = 6): LinkBarTextItem[] => {
  const categories = ['News', 'Reviews', 'Guides', 'Videos', 'Forums', 'Community', 'Events', 'Merchandise'];
  return Array.from({ length: count }, (_, i) => ({
    label: categories[i] || `Category ${i + 1}`,
    route_name: `category_${i}`,
    route_params: {},
  }));
};

const createImageLinks = (count = 5): LinkBarImageItem[] => {
  return Array.from({ length: count }, (_, i) => ({
    label: `Partner ${i + 1}`,
    img: `https://picsum.photos/120/40?random=${i + 1}`,
    alt: `Partner ${i + 1} Logo`,
    width: 120,
    height: 40,
    url: `https://partner${i + 1}.example.com`,
    seoLink: 'obfuscated',
  }));
};

export const Default: Story = {
  args: {
    link_bar: createTextLinks(6),
    link_bar_classes: [],
  },
};

export const ManyLinks: Story = {
  args: {
    link_bar: createTextLinks(12),
    link_bar_classes: [],
  },
};

export const FewLinks: Story = {
  args: {
    link_bar: createTextLinks(3),
    link_bar_classes: [],
  },
};

export const WithImageLogos: Story = {
  args: {
    link_bar: createImageLinks(5),
    link_bar_classes: [],
  },
};

export const MixedTextAndImages: Story = {
  args: {
    link_bar: [
      { label: 'News', route_name: 'news', route_params: {} },
      { 
        label: 'Sponsor 1',
        img: 'https://picsum.photos/100/30?random=sponsor',
        alt: 'Sponsor Logo',
        width: 100,
        height: 30,
        url: 'https://sponsor.example.com',
        seoLink: 'obfuscated',
      },
      { label: 'Reviews', route_name: 'reviews', route_params: {} },
      { label: 'Guides', route_name: 'guides', route_params: {} },
      { 
        label: 'Partner',
        img: 'https://picsum.photos/100/30?random=partner',
        alt: 'Partner Logo',
        width: 100,
        height: 30,
        url: 'https://partner.example.com',
        seoLink: 'obfuscated',
      },
      { label: 'Community', route_name: 'community', route_params: {} },
    ],
    link_bar_classes: [],
  },
};

export const ExternalLinks: Story = {
  args: {
    link_bar: [
      { label: 'YouTube', url: 'https://youtube.com', seoLink: 'obfuscated' },
      { label: 'Twitter', url: 'https://twitter.com', seoLink: 'obfuscated' },
      { label: 'Discord', url: 'https://discord.com', seoLink: 'obfuscated' },
      { label: 'Twitch', url: 'https://twitch.tv', seoLink: 'obfuscated' },
      { label: 'Instagram', url: 'https://instagram.com', seoLink: 'obfuscated' },
    ],
    link_bar_classes: [],
  },
};

export const WithCustomClasses: Story = {
  args: {
    link_bar: createTextLinks(6),
    link_bar_classes: ['custom-link-bar', 'highlighted'],
  },
};

export const WithTracking: Story = {
  args: {
    link_bar: [
      { 
        label: 'Featured News',
        route_name: 'news',
        route_params: {},
        tracking: 'linkbar_news_click',
        data_attr: { 'data-position': '1' },
      },
      { 
        label: 'Hot Reviews',
        route_name: 'reviews',
        route_params: {},
        tracking: 'linkbar_reviews_click',
        data_attr: { 'data-position': '2' },
      },
      { 
        label: 'Latest Guides',
        route_name: 'guides',
        route_params: {},
        tracking: 'linkbar_guides_click',
        data_attr: { 'data-position': '3' },
      },
    ],
    link_bar_classes: [],
  },
};

export const ImageOnlyLinks: Story = {
  args: {
    link_bar: Array.from({ length: 6 }, (_, i) => ({
      label: `Partner ${i + 1}`,
      img: `https://picsum.photos/100/40?random=${i + 10}`,
      alt: `Partner ${i + 1} Logo`,
      width: 100,
      height: 40,
      no_label: true,
      url: `https://partner${i + 1}.example.com`,
      seoLink: 'obfuscated',
    })),
    link_bar_classes: [],
  },
};

export const LongLabels: Story = {
  args: {
    link_bar: [
      { label: 'Breaking News & Updates', route_name: 'news', route_params: {} },
      { label: 'In-Depth Game Reviews', route_name: 'reviews', route_params: {} },
      { label: 'Strategy Guides & Walkthroughs', route_name: 'guides', route_params: {} },
      { label: 'Video Content & Streams', route_name: 'videos', route_params: {} },
      { label: 'Community Forums & Discussion', route_name: 'forums', route_params: {} },
    ],
    link_bar_classes: [],
  },
};

export const WithCustomElementIds: Story = {
  args: {
    link_bar: [
      { label: 'News', route_name: 'news', route_params: {}, element_id: 'linkbar-news' },
      { label: 'Reviews', route_name: 'reviews', route_params: {}, element_id: 'linkbar-reviews' },
      { label: 'Guides', route_name: 'guides', route_params: {}, element_id: 'linkbar-guides' },
    ],
    link_bar_classes: [],
  },
};
