/**
 * Header Main V2 Module Stories
 *
 * Main site header with logo, navigation menu, search, and optional features.
 *
 * This module provides the primary navigation header for websites. It includes:
 * - Logo with configurable dimensions
 * - Hamburger menu toggle
 * - Primary navigation menu (raw menu format or KNP menu)
 * - Search functionality with cancel button
 * - Optional additional buttons (newsletter, dark mode)
 * - Responsive mobile/desktop layouts
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface MenuItem {
  label: string;
  route_name?: string;
  route_params?: Record<string, unknown>;
  url?: string;
  target_blank?: boolean;
  seo_link?: string;
  items?: MenuItem[];
}

interface SearchLink {
  label: string;
  route_name: string;
  route_params: Record<string, unknown>;
}

interface HeaderArgs {
  logo_src: string;
  logo_path: string;
  logo_alt: string;
  logo_width: number;
  logo_height: number;
  menu_items: Record<string, MenuItem>;
  menu_has_subnav_html: boolean;
  search_path: string;
  search_placeholder: string;
  search_cancel_label: string;
  search_input_name: string;
  search_input_id: string;
  search_links: SearchLink[];
  newsletter_href?: string | null;
  dark_mode: boolean;
}

// Component metadata
const meta: Meta<HeaderArgs> = {
  title: 'Modules/Header/Header Main V2',
  parameters: {
    docs: {
      description: {
        component: `Primary site header with logo, navigation, and search functionality.

**Features:**
- Logo with customizable dimensions and alt text
- Mobile hamburger menu with open/close states
- Multi-level navigation menu support
- Integrated search with example links
- Optional newsletter and dark mode toggles
- Responsive design (mobile menu panel)

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/header/header_main_v2.html.twig' with {
    logo_src: '/path/to/logo.svg',
    logo_path: '/',
    logo_alt: 'Site Name',
    menu_obj: menuBuilder.getMenu(),
    search_path: path('search'),
} only %}
\`\`\``,
      },
    },
    twig: {
      template: '_stories/header-main-v2-story.html.twig',
      componentPath: 'module/header/header_main_v2',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    logo_src: {
      control: 'text',
      description: 'Logo image source URL',
      table: { category: 'Logo' },
    },
    logo_path: {
      control: 'text',
      description: 'Logo link destination (typically homepage)',
      table: { category: 'Logo' },
    },
    logo_alt: {
      control: 'text',
      description: 'Logo alt text for accessibility',
      table: { category: 'Logo' },
    },
    logo_width: {
      control: 'number',
      description: 'Logo width in pixels',
      table: { category: 'Logo' },
    },
    logo_height: {
      control: 'number',
      description: 'Logo height in pixels',
      table: { category: 'Logo' },
    },
    menu_items: {
      control: 'object',
      description: 'Navigation menu items array',
      table: { category: 'Navigation' },
    },
    menu_has_subnav_html: {
      control: 'boolean',
      description: 'Render subnav items as HTML (vs lazy-loaded)',
      table: { category: 'Navigation' },
    },
    search_path: {
      control: 'text',
      description: 'Search form action URL',
      table: { category: 'Search' },
    },
    search_placeholder: {
      control: 'text',
      description: 'Search input placeholder text',
      table: { category: 'Search' },
    },
    search_cancel_label: {
      control: 'text',
      description: 'Search cancel button text',
      table: { category: 'Search' },
    },
    search_input_name: {
      control: 'text',
      description: 'Search input field name attribute',
      table: { category: 'Search' },
    },
    search_input_id: {
      control: 'text',
      description: 'Search input field ID attribute',
      table: { category: 'Search' },
    },
    search_links: {
      control: 'object',
      description: 'Example search links to display',
      table: { category: 'Search' },
    },
    newsletter_href: {
      control: 'text',
      description: 'Newsletter subscription link',
      table: { category: 'Features' },
    },
    dark_mode: {
      control: 'boolean',
      description: 'Show dark mode toggle button',
      table: { category: 'Features' },
    },
  },
};

export default meta;
type Story = StoryObj<HeaderArgs>;

// Mock menu data helper
const createMockMenu = (includeSubnav = false): Record<string, MenuItem> => {
  const menu: Record<string, MenuItem> = {
    home: {
      label: 'Home',
      route_name: 'home',
      route_params: {},
    },
    news: {
      label: 'News',
      route_name: 'news',
      route_params: {},
    },
    reviews: {
      label: 'Reviews',
      route_name: 'reviews',
      route_params: {},
    },
    videos: {
      label: 'Videos',
      route_name: 'videos',
      route_params: {},
    },
    guides: {
      label: 'Guides',
      route_name: 'guides',
      route_params: {},
    },
  };

  if (includeSubnav) {
    menu.news.items = [
      { label: 'Breaking News', route_name: 'news_breaking', route_params: {} },
      { label: 'Updates', route_name: 'news_updates', route_params: {} },
      { label: 'Patches', route_name: 'news_patches', route_params: {} },
    ];
    menu.reviews.items = [
      { label: 'Game Reviews', route_name: 'reviews_games', route_params: {} },
      { label: 'Hardware Reviews', route_name: 'reviews_hardware', route_params: {} },
    ];
  }

  return menu;
};

// Stories
export const Default: Story = {
  args: {
    logo_src: 'https://picsum.photos/200/60',
    logo_path: '/',
    logo_alt: 'Site Logo',
    logo_width: 200,
    logo_height: 60,
    menu_items: createMockMenu(false),
    menu_has_subnav_html: false,
    search_path: '/search',
    search_placeholder: 'Search the website...',
    search_cancel_label: 'Cancel',
    search_input_name: 'q',
    search_input_id: 'search_bar',
    search_links: [],
    newsletter_href: null,
    dark_mode: false,
  },
};

export const WithSubnavigation: Story = {
  args: {
    ...Default.args,
    menu_items: createMockMenu(true),
    menu_has_subnav_html: true,
  },
};

export const WithSearchExamples: Story = {
  args: {
    ...Default.args,
    search_links: [
      { label: 'Zelda', route_name: 'search_zelda', route_params: {} },
      { label: 'Elden Ring', route_name: 'search_elden', route_params: {} },
      { label: 'Pokemon', route_name: 'search_pokemon', route_params: {} },
    ],
  },
};

export const WithAllFeatures: Story = {
  args: {
    ...Default.args,
    menu_items: createMockMenu(true),
    menu_has_subnav_html: true,
    newsletter_href: '/newsletter',
    dark_mode: true,
    search_links: [
      { label: 'Trending', route_name: 'search_trending', route_params: {} },
      { label: 'Popular', route_name: 'search_popular', route_params: {} },
    ],
  },
};

export const MinimalHeader: Story = {
  args: {
    logo_src: 'https://picsum.photos/150/50',
    logo_path: '/',
    logo_alt: 'Minimal Logo',
    logo_width: 150,
    logo_height: 50,
    menu_items: {
      home: { label: 'Home', route_name: 'home', route_params: {} },
      about: { label: 'About', route_name: 'about', route_params: {} },
      contact: { label: 'Contact', route_name: 'contact', route_params: {} },
    },
    menu_has_subnav_html: false,
    search_path: '/search',
    search_placeholder: 'Search...',
    search_cancel_label: 'Cancel',
    search_input_name: 'q',
    search_input_id: 'search_bar',
    search_links: [],
    newsletter_href: null,
    dark_mode: false,
  },
};

export const LongMenuItems: Story = {
  args: {
    ...Default.args,
    menu_items: {
      home: { label: 'Home', route_name: 'home', route_params: {} },
      'breaking-news': { label: 'Breaking News', route_name: 'breaking', route_params: {} },
      'game-reviews': { label: 'Game Reviews', route_name: 'reviews', route_params: {} },
      'video-content': { label: 'Video Content', route_name: 'videos', route_params: {} },
      'strategy-guides': { label: 'Strategy Guides', route_name: 'guides', route_params: {} },
      community: { label: 'Community', route_name: 'community', route_params: {} },
      forums: { label: 'Forums', route_name: 'forums', route_params: {} },
      merchandise: { label: 'Merchandise', route_name: 'merch', route_params: {} },
    },
  },
};

export const WithExternalLinks: Story = {
  args: {
    ...Default.args,
    menu_items: {
      home: { label: 'Home', route_name: 'home', route_params: {} },
      news: { label: 'News', route_name: 'news', route_params: {} },
      youtube: {
        label: 'YouTube',
        url: 'https://youtube.com',
        target_blank: true,
        seo_link: 'obfuscated',
      },
      discord: {
        label: 'Discord',
        url: 'https://discord.com',
        target_blank: true,
        seo_link: 'obfuscated',
      },
    },
  },
};

export const DarkModeEnabled: Story = {
  args: {
    ...Default.args,
    dark_mode: true,
    logo_src: 'https://picsum.photos/200/60',
  },
};

export const NewsletterSubscription: Story = {
  args: {
    ...Default.args,
    newsletter_href: 'https://example.com/newsletter',
  },
};
