/**
 * Dropdown Module Stories
 *
 * Dropdown/select component for filtering and navigation.
 * 
 * This module displays a button that expands to show a list of options.
 * It supports filtered states where a "clear" action is shown instead of the dropdown.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface DropdownData {
  name: string;
  url: string;
  is_current: boolean;
}

interface DropdownArgs {
  cta_text: string;
  already_filtered?: boolean;
  default_path?: string;
  dropdown_classes?: string[];
  value_key?: string;
  current_status?: string;
  data?: DropdownData[];
}

// Component metadata
const meta: Meta<DropdownArgs> = {
  title: 'Modules/Dropdown',
  parameters: {
    docs: {
      description: {
        component: 'Dropdown component for filtering and navigation with support for active states.',
      },
    },
    twig: {
      template: '_stories/dropdown-story.html.twig',
      componentPath: 'module/dropdown/dropdown',
      argsToContext: (args) => ({
        dropdown_classes: args.dropdown_classes,
        already_filtered: args.already_filtered,
        default_path: args.default_path,
        cta_text: args.cta_text,
        data: args.data,
        current_status: args.current_status,
        value_key: args.value_key,
      }),
    },
  } as StoryParameters,
  argTypes: {
    cta_text: {
      control: 'text',
      description: 'Text displayed on the dropdown button',
      table: { category: 'Content' },
    },
    already_filtered: {
      control: 'boolean',
      description: 'Whether dropdown is in filtered state (shows clear button)',
      table: { category: 'State' },
    },
    default_path: {
      control: 'text',
      description: 'URL to redirect when clearing filter',
      table: { category: 'Navigation' },
      if: { arg: 'already_filtered', truthy: true },
    },
    dropdown_classes: {
      control: 'object',
      description: 'Additional CSS classes for dropdown container',
      table: { category: 'Style' },
    },
    value_key: {
      control: 'text',
      description: 'Key to access display value in data items',
      table: { category: 'Data' },
    },
    current_status: {
      control: 'text',
      description: 'Key to check if item is currently selected',
      table: { category: 'Data' },
    },
  },
};

export default meta;
type Story = StoryObj<DropdownArgs>;

// Stories
export const Default: Story = {
  args: {
    cta_text: 'Select Category',
    already_filtered: false,
    dropdown_classes: [],
    value_key: 'name',
    current_status: 'is_current',
    data: [
      { name: 'Technology', url: '/category/tech', is_current: false },
      { name: 'Gaming', url: '/category/gaming', is_current: false },
      { name: 'Entertainment', url: '/category/entertainment', is_current: false },
      { name: 'Sports', url: '/category/sports', is_current: false },
      { name: 'News', url: '/category/news', is_current: false },
    ],
  },
};

export const WithSelectedItem: Story = {
  args: {
    ...Default.args,
    data: [
      { name: 'Technology', url: '/category/tech', is_current: false },
      { name: 'Gaming', url: '/category/gaming', is_current: true },
      { name: 'Entertainment', url: '/category/entertainment', is_current: false },
      { name: 'Sports', url: '/category/sports', is_current: false },
      { name: 'News', url: '/category/news', is_current: false },
    ],
  },
};

export const FilteredState: Story = {
  args: {
    cta_text: 'Gaming',
    already_filtered: true,
    default_path: '/categories',
    dropdown_classes: ['filtered'],
    data: [],
  },
};

export const LongList: Story = {
  args: {
    cta_text: 'Choose a Country',
    already_filtered: false,
    dropdown_classes: [],
    value_key: 'name',
    current_status: 'is_current',
    data: [
      { name: 'France', url: '/country/fr', is_current: false },
      { name: 'United States', url: '/country/us', is_current: false },
      { name: 'United Kingdom', url: '/country/uk', is_current: false },
      { name: 'Germany', url: '/country/de', is_current: false },
      { name: 'Spain', url: '/country/es', is_current: false },
      { name: 'Italy', url: '/country/it', is_current: false },
      { name: 'Canada', url: '/country/ca', is_current: false },
      { name: 'Australia', url: '/country/au', is_current: false },
      { name: 'Japan', url: '/country/jp', is_current: false },
      { name: 'Brazil', url: '/country/br', is_current: false },
    ],
  },
};

export const WithCustomClasses: Story = {
  args: {
    ...Default.args,
    dropdown_classes: ['dropdown-primary', 'dropdown-large'],
  },
};
