/**
 * Directory Alpha Module Stories
 *
 * Alphabetical directory listing with letter header and entity links.
 * 
 * This module displays an alphabetical directory section with a letter header
 * and a list of entities starting with that letter. Commonly used for A-Z indexes,
 * author directories, game lists, etc.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface Entity {
  id: number;
  name: string;
  slug: string;
}

interface DirectoryAlphaArgs {
  letter_directory: string;
  entities_directory: Entity[];
  title_markup: 'span' | 'div' | 'h2' | 'h3' | 'h4';
  item_markup: 'span' | 'div' | 'li';
  cta_all_enable: boolean;
  cta_all_txt: string;
  directory_alpha_classes: string[];
}

const meta: Meta<DirectoryAlphaArgs> = {
  title: 'Modules/Directory Alpha',
  parameters: {
    docs: {
      description: {
        component: `Alphabetical directory component for A-Z listings.
        
**Features:**
- Letter header display
- Entity name links
- Optional "View All" CTA
- Customizable HTML tags (span/div/etc)
- Block overrides for href/text
- Custom CSS classes

**Usage in projects:**
\`\`\`twig
{% embed '@Ui/module/directory/directory_alpha.html.twig' with {
    letter_directory: 'A',
    entities_directory: entities,
    cta_all_enable: true,
    cta_all_txt: 'View all A'
} %}
    {% block entity_directory_item_href %}{{ path('entity', {id: entity_directory.id}) }}{% endblock %}
    {% block entity_directory_item_txt %}{{ entity_directory.name }}{% endblock %}
{% endembed %}
\`\`\``,
      },
    },
    twig: {
      template: '_stories/directory-alpha-story.html.twig',
      componentPath: 'module/directory/directory_alpha',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    letter_directory: {
      control: 'text',
      description: 'Letter header to display',
      table: { category: 'Content' },
    },
    entities_directory: {
      control: 'object',
      description: 'Array of entities for this letter',
      table: { category: 'Content' },
    },
    title_markup: {
      control: 'select',
      options: ['span', 'div', 'h2', 'h3', 'h4'],
      description: 'HTML tag for letter header',
      table: { category: 'Semantic' },
    },
    item_markup: {
      control: 'select',
      options: ['span', 'div', 'li'],
      description: 'HTML tag for entity items',
      table: { category: 'Semantic' },
    },
    cta_all_enable: {
      control: 'boolean',
      description: 'Show "View All" CTA',
      table: { category: 'CTA' },
    },
    cta_all_txt: {
      control: 'text',
      description: '"View All" CTA text',
      table: { category: 'CTA' },
    },
    directory_alpha_classes: {
      control: 'object',
      description: 'Additional CSS classes',
      table: { category: 'Style' },
    },
  },
};

export default meta;
type Story = StoryObj<DirectoryAlphaArgs>;

// Mock data helpers
const createEntities = (letter: string, count = 5): Entity[] => {
  const names: Record<string, string[]> = {
    A: ['Assassin\'s Creed', 'Animal Crossing', 'Apex Legends', 'Among Us', 'Astral Chain'],
    B: ['Baldur\'s Gate', 'Bloodborne', 'Battlefield', 'Bayonetta', 'Borderlands'],
    C: ['Cyberpunk 2077', 'Call of Duty', 'Cuphead', 'Celeste', 'Control'],
    D: ['Dark Souls', 'Destiny', 'DOOM', 'Dead Space', 'Diablo'],
    Z: ['Zelda', 'Zenless Zone Zero', 'Zombie Army'],
  };
  
  const list = names[letter] || [`${letter}ame 1`, `${letter}ame 2`, `${letter}ame 3`];
  return list.slice(0, count).map((name, i) => ({
    id: i + 1,
    name: name,
    slug: name.toLowerCase().replace(/[^a-z0-9]+/g, '-'),
  }));
};

export const LetterA: Story = {
  args: {
    letter_directory: 'A',
    entities_directory: createEntities('A', 5),
    title_markup: 'span',
    item_markup: 'span',
    cta_all_enable: false,
    cta_all_txt: '',
    directory_alpha_classes: [],
  },
};

export const WithViewAllCTA: Story = {
  args: {
    letter_directory: 'B',
    entities_directory: createEntities('B', 5),
    title_markup: 'span',
    item_markup: 'span',
    cta_all_enable: true,
    cta_all_txt: 'View all B games',
    directory_alpha_classes: [],
  },
};

export const FewEntities: Story = {
  args: {
    letter_directory: 'Z',
    entities_directory: createEntities('Z', 3),
    title_markup: 'span',
    item_markup: 'span',
    cta_all_enable: false,
    cta_all_txt: '',
    directory_alpha_classes: [],
  },
};

export const ManyEntities: Story = {
  args: {
    letter_directory: 'C',
    entities_directory: [
      ...createEntities('C', 5),
      { id: 6, name: 'Crash Bandicoot', slug: 'crash-bandicoot' },
      { id: 7, name: 'Civilization', slug: 'civilization' },
      { id: 8, name: 'Cities Skylines', slug: 'cities-skylines' },
      { id: 9, name: 'Chivalry', slug: 'chivalry' },
      { id: 10, name: 'Crysis', slug: 'crysis' },
    ],
    title_markup: 'span',
    item_markup: 'span',
    cta_all_enable: true,
    cta_all_txt: 'View all C games',
    directory_alpha_classes: [],
  },
};

export const WithH3Header: Story = {
  args: {
    letter_directory: 'D',
    entities_directory: createEntities('D', 5),
    title_markup: 'h3',
    item_markup: 'span',
    cta_all_enable: false,
    cta_all_txt: '',
    directory_alpha_classes: [],
  },
};

export const WithListItems: Story = {
  args: {
    letter_directory: 'A',
    entities_directory: createEntities('A', 5),
    title_markup: 'h3',
    item_markup: 'li',
    cta_all_enable: true,
    cta_all_txt: 'See all',
    directory_alpha_classes: [],
  },
};

export const CustomClasses: Story = {
  args: {
    letter_directory: 'B',
    entities_directory: createEntities('B', 5),
    title_markup: 'span',
    item_markup: 'span',
    cta_all_enable: false,
    cta_all_txt: '',
    directory_alpha_classes: ['featured-directory', 'highlighted'],
  },
};

export const NoEntities: Story = {
  args: {
    letter_directory: 'X',
    entities_directory: [],
    title_markup: 'span',
    item_markup: 'span',
    cta_all_enable: false,
    cta_all_txt: '',
    directory_alpha_classes: [],
  },
};
