/**
 * Breadcrumb Module Stories
 *
 * Hierarchical navigation breadcrumb with H1 title.
 * 
 * This module displays a breadcrumb navigation trail followed by an H1 heading.
 * The breadcrumb and H1 are typically populated via blocks from consuming projects,
 * often using SEO bundle data.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface BreadcrumbArgs {
  seo_breadcrumb: string;
  seo_h1_title: string;
  aria_level?: '1' | '2' | '3' | '4' | '5' | '6';
}

// Component metadata
const meta: Meta<BreadcrumbArgs> = {
  title: 'Modules/Navigation/Breadcrumb',
  parameters: {
    docs: {
      description: {
        component: `Breadcrumb navigation with integrated H1 heading for page hierarchy.
        
**Features:**
- Hierarchical breadcrumb navigation
- Integrated H1 heading with configurable aria-level
- Spaceless rendering for clean markup
- Block overrides for customization
- SEO-friendly structure

**Usage in projects:**
\`\`\`twig
{% embed '@Ui/module/breadcrumb/breadcrumb.html.twig' with {
    seo_breadcrumb: '<a href="/">Home</a> > <a href="/news">News</a>',
    seo_h1_title: 'Article Title',
    aria_level: '1'
} %}
    {# Optional: Override breadcrumb or h1Title blocks #}
{% endembed %}
\`\`\`

**Note:** Breadcrumb HTML is typically generated by SEO bundle or similar service.`,
      },
    },
    twig: {
      template: '_stories/breadcrumb-story.html.twig',
      componentPath: 'module/breadcrumb/breadcrumb',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    seo_breadcrumb: {
      control: 'text',
      description: 'Breadcrumb HTML markup',
      table: { category: 'Content' },
    },
    seo_h1_title: {
      control: 'text',
      description: 'Page H1 title text',
      table: { category: 'Content' },
    },
    aria_level: {
      control: 'select',
      options: ['1', '2', '3', '4', '5', '6'],
      description: 'ARIA heading level for H1',
      table: { category: 'Accessibility' },
    },
  },
};

export default meta;
type Story = StoryObj<BreadcrumbArgs>;

// Stories
export const Default: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/news" class="item">News</a>',
    seo_h1_title: 'Breaking: New Game Announcement',
    aria_level: '1',
  },
};

export const ShortPath: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a>',
    seo_h1_title: 'Home Page',
    aria_level: '1',
  },
};

export const DeepPath: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/games" class="item">Games</a> <a href="/games/rpg" class="item">RPG</a> <a href="/games/rpg/reviews" class="item">Reviews</a>',
    seo_h1_title: 'Final Fantasy XVI Review',
    aria_level: '1',
  },
};

export const CategoryPage: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/reviews" class="item">Reviews</a>',
    seo_h1_title: 'Game Reviews',
    aria_level: '1',
  },
};

export const ArticlePage: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/news" class="item">News</a> <a href="/news/gaming" class="item">Gaming</a>',
    seo_h1_title: 'The Future of Cloud Gaming: What to Expect in 2024',
    aria_level: '1',
  },
};

export const GuidePage: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/guides" class="item">Guides</a> <a href="/guides/zelda" class="item">Zelda</a>',
    seo_h1_title: 'Complete Walkthrough: Tears of the Kingdom',
    aria_level: '1',
  },
};

export const LongTitle: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/features" class="item">Features</a>',
    seo_h1_title: 'An In-Depth Analysis of Game Design Principles: How Modern RPGs Balance Story and Gameplay Mechanics',
    aria_level: '1',
  },
};

export const WithDifferentAriaLevel: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a> <a href="/news" class="item">News</a>',
    seo_h1_title: 'Latest Gaming News',
    aria_level: '2',
  },
};

export const ReviewPage: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a><a href="/reviews" class="item">Reviews</a><a href="/reviews/ps5" class="item">PS5</a>',
    seo_h1_title: 'Spider-Man 2 Review',
    aria_level: '1',
  },
};

export const PokemonGuide: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a><a href="/guides" class="item">Guides</a><a href="/guides/pokemon" class="item">Pokemon</a>',
    seo_h1_title: 'Pokemon Scarlet & Violet Team Builder',
    aria_level: '1',
  },
};

export const SingleLevel: Story = {
  args: {
    seo_breadcrumb: '<a href="/" class="item">Home</a>',
    seo_h1_title: 'All News Articles',
    aria_level: '1',
  },
};
