/**
 * Author Description Module Stories
 *
 * Author profile card with photo, name, role, and social media links.
 * 
 * This module displays a complete author profile card featuring:
 * - Circular profile photo
 * - Author name and job title
 * - Social media links (X/Twitter, LinkedIn)
 * - Clean, card-based layout
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface AuthorDescriptionArgs {
  author_name: string;
  author_job: string | null;
  author_img: string | null;
  author_hasSocialUrls: boolean;
  author_x_url?: string;
  author_x_name?: string;
  author_x_logo?: string;
  author_x_logo_width?: number;
  author_x_logo_height?: number;
  author_linkedin_Url?: string;
  author_linkedin_name?: string;
  author_linkedin_logo?: string;
  author_linkedin_logo_width?: number;
  author_linkedin_logo_height?: number;
}

const meta: Meta<AuthorDescriptionArgs> = {
  title: 'Modules/Author/Author Description',
  parameters: {
    docs: {
      description: {
        component: `Author profile card with photo, role, and social network links.
        
**Features:**
- Circular author avatar (105x105px)
- Author name display
- Job title/role
- Social media integration (X/Twitter, LinkedIn)
- Social network logos
- Clean card layout

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/author/author_description.html.twig' with {
    author_name: 'Jane Smith',
    author_job: 'Senior Editor',
    author_img: '/path/to/avatar.jpg',
    author_hasSocialUrls: true,
    author_x_url: 'https://x.com/janesmith'
} only %}
\`\`\``,
      },
    },
    twig: {
      template: 'module/author/author_description.html.twig',
      renderMode: 'include',
      componentPath: 'module/author/author_description',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    author_name: {
      control: 'text',
      description: 'Author full name',
      table: { category: 'Content' },
    },
    author_job: {
      control: 'text',
      description: 'Author job title or role',
      table: { category: 'Content' },
    },
    author_img: {
      control: 'text',
      description: 'Author profile image URL',
      table: { category: 'Media' },
    },
    author_hasSocialUrls: {
      control: 'boolean',
      description: 'Show social media section',
      table: { category: 'Social' },
    },
    author_x_url: {
      control: 'text',
      description: 'X (Twitter) profile URL',
      table: { category: 'Social' },
    },
    author_x_name: {
      control: 'text',
      description: 'X username (without @)',
      table: { category: 'Social' },
    },
    author_x_logo: {
      control: 'text',
      description: 'X logo image URL',
      table: { category: 'Social' },
    },
    author_linkedin_Url: {
      control: 'text',
      description: 'LinkedIn profile URL',
      table: { category: 'Social' },
    },
    author_linkedin_name: {
      control: 'text',
      description: 'LinkedIn display name',
      table: { category: 'Social' },
    },
    author_linkedin_logo: {
      control: 'text',
      description: 'LinkedIn logo image URL',
      table: { category: 'Social' },
    },
  },
};

export default meta;
type Story = StoryObj<AuthorDescriptionArgs>;

export const Default: Story = {
  args: {
    author_name: 'Jane Smith',
    author_job: 'Senior Gaming Editor',
    author_img: 'https://picsum.photos/105/105?random=1',
    author_hasSocialUrls: false,
  },
};

export const WithTwitter: Story = {
  args: {
    author_name: 'John Doe',
    author_job: 'Reviews Editor',
    author_img: 'https://picsum.photos/105/105?random=2',
    author_hasSocialUrls: true,
    author_x_url: 'https://x.com/johndoe',
    author_x_name: 'johndoe',
    author_x_logo: 'https://picsum.photos/20/20?random=x',
    author_x_logo_width: 20,
    author_x_logo_height: 20,
  },
};

export const WithLinkedIn: Story = {
  args: {
    author_name: 'Sarah Martinez',
    author_job: 'Industry Analyst',
    author_img: 'https://picsum.photos/105/105?random=3',
    author_hasSocialUrls: true,
    author_linkedin_Url: 'https://linkedin.com/in/sarahmartinez',
    author_linkedin_name: 'Sarah Martinez',
    author_linkedin_logo: 'https://picsum.photos/20/20?random=li',
    author_linkedin_logo_width: 20,
    author_linkedin_logo_height: 20,
  },
};

export const WithBothSocials: Story = {
  args: {
    author_name: 'Mike Chen',
    author_job: 'Lead Writer',
    author_img: 'https://picsum.photos/105/105?random=4',
    author_hasSocialUrls: true,
    author_x_url: 'https://x.com/mikechen',
    author_x_name: 'mikechen',
    author_x_logo: 'https://picsum.photos/20/20?random=x',
    author_x_logo_width: 20,
    author_x_logo_height: 20,
    author_linkedin_Url: 'https://linkedin.com/in/mikechen',
    author_linkedin_name: 'Mike Chen',
    author_linkedin_logo: 'https://picsum.photos/20/20?random=li',
    author_linkedin_logo_width: 20,
    author_linkedin_logo_height: 20,
  },
};

export const NoImage: Story = {
  args: {
    author_name: 'Anonymous Writer',
    author_job: 'Contributor',
    author_img: null,
    author_hasSocialUrls: false,
  },
};

export const LongJobTitle: Story = {
  args: {
    author_name: 'Dr. Alexandra Thompson',
    author_job: 'Senior Technology Correspondent & Gaming Industry Analyst',
    author_img: 'https://picsum.photos/105/105?random=5',
    author_hasSocialUrls: false,
  },
};

export const FreelanceContributor: Story = {
  args: {
    author_name: 'Chris Taylor',
    author_job: 'Freelance Gaming Journalist',
    author_img: 'https://picsum.photos/105/105?random=6',
    author_hasSocialUrls: true,
    author_x_url: 'https://x.com/christaylor',
    author_x_name: 'christaylor',
    author_x_logo: 'https://picsum.photos/20/20?random=x',
    author_x_logo_width: 20,
    author_x_logo_height: 20,
    author_linkedin_Url: 'https://linkedin.com/in/christaylor',
    author_linkedin_name: 'Chris Taylor',
    author_linkedin_logo: 'https://picsum.photos/20/20?random=li',
    author_linkedin_logo_width: 20,
    author_linkedin_logo_height: 20,
  },
};

export const EditorInChief: Story = {
  args: {
    author_name: 'Rachel Green',
    author_job: 'Editor in Chief',
    author_img: 'https://picsum.photos/105/105?random=7',
    author_hasSocialUrls: true,
    author_x_url: 'https://x.com/rachelgreen',
    author_x_name: 'rachelgreen',
    author_x_logo: 'https://picsum.photos/20/20?random=x',
    author_x_logo_width: 20,
    author_x_logo_height: 20,
  },
};

export const WithLongName: Story = {
  args: {
    author_name: 'Dr. Elizabeth Anne Montgomery-Williams',
    author_job: 'Gaming Psychologist',
    author_img: 'https://picsum.photos/105/105?random=8',
    author_hasSocialUrls: true,
    author_linkedin_Url: 'https://linkedin.com/in/montgomery-williams',
    author_linkedin_name: 'Elizabeth Montgomery-Williams',
    author_linkedin_logo: 'https://picsum.photos/20/20?random=li',
    author_linkedin_logo_width: 20,
    author_linkedin_logo_height: 20,
  },
};

export const MinimalProfile: Story = {
  args: {
    author_name: 'Guest',
    author_job: null,
    author_img: 'https://picsum.photos/105/105?random=9',
    author_hasSocialUrls: false,
  },
};
