/**
 * Author Biography Module Stories
 *
 * Author information display with avatar, bio, and co-authors.
 * 
 * This module displays comprehensive author information including profile picture,
 * name, status/role, biography text, and optional co-authors. Supports two display
 * types: short bio for article pages or standalone bio block.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface CoAuthor {
  name: string;
  id: number;
  slug: string;
  isBrowsable: boolean;
}

interface AuthorBiographyArgs {
  type: 'author-article-short-bio' | 'standalone';
  author_name: string;
  author_status: string;
  author_bio: string;
  author_img: string;
  author_img_width: number;
  author_img_height: number;
  author_link: string;
  author_isEnabled: boolean;
  by: string;
  status_separator: string;
  co_authors: CoAuthor[];
  co_author_by?: string;
}

const meta: Meta<AuthorBiographyArgs> = {
  title: 'Modules/Author/Author Biography',
  parameters: {
    docs: {
      description: {
        component: `Author biography component with avatar, role, and description.
        
**Features:**
- Author profile image
- Author name with optional link
- Role/status display
- Biography text (raw HTML)
- Co-authors listing
- Two display types: article short bio or standalone
- Social media links support

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/author/author_biography.html.twig' with {
    type: 'author-article-short-bio',
    author_name: 'John Doe',
    author_status: 'Senior Writer',
    author_bio: '<p>Bio text...</p>'
} only %}
\`\`\``,
      },
    },
    twig: {
      template: '_stories/author-biography-story.html.twig',
      componentPath: 'module/author/author_biography',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    type: {
      control: 'select',
      options: ['author-article-short-bio', 'standalone'],
      description: 'Display type',
      table: { category: 'Layout' },
    },
    author_name: {
      control: 'text',
      description: 'Author full name',
      table: { category: 'Content' },
    },
    author_status: {
      control: 'text',
      description: 'Author role or status',
      table: { category: 'Content' },
    },
    author_bio: {
      control: 'text',
      description: 'Author biography HTML',
      table: { category: 'Content' },
    },
    author_img: {
      control: 'text',
      description: 'Author profile image URL',
      table: { category: 'Media' },
    },
    author_img_width: {
      control: 'number',
      description: 'Profile image width',
      table: { category: 'Media' },
    },
    author_img_height: {
      control: 'number',
      description: 'Profile image height',
      table: { category: 'Media' },
    },
    author_link: {
      control: 'text',
      description: 'Link to author profile page',
      table: { category: 'Navigation' },
    },
    author_isEnabled: {
      control: 'boolean',
      description: 'Enable author profile link',
      table: { category: 'Navigation' },
    },
    by: {
      control: 'text',
      description: '"By" prefix text',
      table: { category: 'Labels' },
    },
    status_separator: {
      control: 'text',
      description: 'Separator between name and status',
      table: { category: 'Labels' },
    },
    co_authors: {
      control: 'object',
      description: 'Array of co-author objects',
      table: { category: 'Content' },
    },
    co_author_by: {
      control: 'text',
      description: 'Co-authors prefix text',
      table: { category: 'Labels' },
    },
  },
};

export default meta;
type Story = StoryObj<AuthorBiographyArgs>;

export const ArticleShortBio: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'Jane Smith',
    author_status: 'Senior Gaming Editor',
    author_bio: '<p>Jane has been covering the gaming industry for over 10 years, specializing in RPGs and narrative-driven experiences.</p>',
    author_img: 'https://picsum.photos/35/35?random=1',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/jane-smith',
    author_isEnabled: true,
    by: 'By',
    status_separator: '|',
    co_authors: [],
    co_author_by: 'With',
  },
};

export const WithCoAuthors: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'John Doe',
    author_status: 'Editor in Chief',
    author_bio: '<p>John leads our editorial team with over 15 years of experience in games journalism.</p>',
    author_img: 'https://picsum.photos/35/35?random=2',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/john-doe',
    author_isEnabled: true,
    by: 'By',
    status_separator: '|',
    co_authors: [
      { name: 'Alice Johnson', id: 1, slug: 'alice-johnson', isBrowsable: true },
      { name: 'Bob Williams', id: 2, slug: 'bob-williams', isBrowsable: true },
    ],
    co_author_by: 'With contributions from',
  },
};

export const NoBiography: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'Mike Chen',
    author_status: 'Staff Writer',
    author_bio: '',
    author_img: 'https://picsum.photos/35/35?random=3',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/mike-chen',
    author_isEnabled: true,
    by: 'By',
    status_separator: '|',
    co_authors: [],
  },
};

export const NoAuthorLink: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'Guest Contributor',
    author_status: 'Guest Writer',
    author_bio: '<p>This article was contributed by a guest writer from the gaming community.</p>',
    author_img: 'https://picsum.photos/35/35?random=4',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '',
    author_isEnabled: false,
    by: 'By',
    status_separator: '|',
    co_authors: [],
  },
};

export const StandaloneBio: Story = {
  args: {
    type: 'standalone',
    author_name: 'Not Used',
    author_status: 'Not Used',
    author_bio: '<p style="padding: 20px; background: #f3f4f6; border-radius: 8px;">This is a standalone bio block without the full author card layout. It displays only the biography content provided.</p>',
    author_img: '',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '',
    author_isEnabled: false,
    by: 'By',
    status_separator: '|',
    co_authors: [],
  },
};

export const LongBiography: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'Sarah Martinez',
    author_status: 'Reviews Editor',
    author_bio: '<p>Sarah Martinez is an award-winning games journalist with a passion for uncovering hidden gems in the indie scene. With a background in computer science and creative writing, she brings a unique perspective to game analysis. When not reviewing games, Sarah streams on Twitch and hosts a popular gaming podcast. Her work has been featured in numerous gaming publications, and she regularly speaks at industry conferences about games criticism and accessibility in gaming.</p>',
    author_img: 'https://picsum.photos/35/35?random=5',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/sarah-martinez',
    author_isEnabled: true,
    by: 'By',
    status_separator: '|',
    co_authors: [],
  },
};

export const MultipleCoAuthors: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'David Park',
    author_status: 'Lead Investigator',
    author_bio: '<p>David leads our investigative journalism team, uncovering stories that matter in the gaming industry.</p>',
    author_img: 'https://picsum.photos/35/35?random=6',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/david-park',
    author_isEnabled: true,
    by: 'By',
    status_separator: '|',
    co_authors: [
      { name: 'Emma Wilson', id: 1, slug: 'emma-wilson', isBrowsable: true },
      { name: 'Tom Anderson', id: 2, slug: 'tom-anderson', isBrowsable: true },
      { name: 'Lisa Brown', id: 3, slug: 'lisa-brown', isBrowsable: true },
      { name: 'Chris Taylor', id: 4, slug: 'chris-taylor', isBrowsable: true },
    ],
    co_author_by: 'With',
  },
};

export const CustomSeparator: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'Rachel Green',
    author_status: 'Contributing Writer',
    author_bio: '<p>Rachel specializes in mobile gaming and esports coverage.</p>',
    author_img: 'https://picsum.photos/35/35?random=7',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/rachel-green',
    author_isEnabled: true,
    by: 'Written by',
    status_separator: '•',
    co_authors: [],
  },
};

export const WithHTMLBio: Story = {
  args: {
    type: 'author-article-short-bio',
    author_name: 'Kevin Zhao',
    author_status: 'Tech Analyst',
    author_bio: '<p><strong>Kevin Zhao</strong> is a technology analyst specializing in gaming hardware and cloud gaming platforms. Follow him on <a href="https://twitter.com" style="color: #2563eb;">Twitter</a> for daily tech insights.</p>',
    author_img: 'https://picsum.photos/35/35?random=8',
    author_img_width: 35,
    author_img_height: 35,
    author_link: '/authors/kevin-zhao',
    author_isEnabled: true,
    by: 'By',
    status_separator: '|',
    co_authors: [],
  },
};
