/**
 * Article Title Module Stories
 *
 * Complete article title section with labels, title, date, and share button.
 * 
 * This module provides a comprehensive article title component that includes:
 * - Category/topic labels
 * - Article title (H1)
 * - Publication and update dates
 * - Sponsored content indicator
 * - Social share button
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface LabelObject {
  text: string;
  type: string;
}

interface ArticleTitleArgs {
  article_title: string;
  label: LabelObject[];
  published_date: string | null;
  updated_date: string | null;
  published_at: string;
  updated_at: string;
  sponso: boolean;
  sponso_text: string;
  share: string | null;
  use_timed_date: boolean;
}

const meta: Meta<ArticleTitleArgs> = {
  title: 'Modules/Article/Article Title',
  parameters: {
    docs: {
      description: {
        component: `Full article title section with metadata and sharing functionality.
        
**Features:**
- Category/topic labels
- SEO-friendly H1 title
- Published and updated date display
- Sponsored content badge
- Social share integration
- Semantic time elements
- Customizable via blocks

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/article/article_title.html.twig' with {
    article_title: 'Article Headline',
    label: [{ text: 'News', type: 'category' }],
    published_date: '2024-01-15',
    share: 'Share'
} only %}
\`\`\``,
      },
    },
    twig: {
      template: 'module/article/article_title.html.twig',
      renderMode: 'include',
      componentPath: 'module/article/article_title',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    article_title: {
      control: 'text',
      description: 'Article title text (H1)',
      table: { category: 'Content' },
    },
    label: {
      control: 'object',
      description: 'Array of category/topic labels',
      table: { category: 'Content' },
    },
    published_date: {
      control: 'text',
      description: 'Publication date',
      table: { category: 'Meta' },
    },
    updated_date: {
      control: 'text',
      description: 'Last update date',
      table: { category: 'Meta' },
    },
    published_at: {
      control: 'text',
      description: 'Published date label text',
      table: { category: 'Meta' },
    },
    updated_at: {
      control: 'text',
      description: 'Updated date label text',
      table: { category: 'Meta' },
    },
    sponso: {
      control: 'boolean',
      description: 'Mark as sponsored content',
      table: { category: 'Meta' },
    },
    sponso_text: {
      control: 'text',
      description: 'Sponsored content text',
      table: { category: 'Meta' },
    },
    share: {
      control: 'text',
      description: 'Share button text (null to hide)',
      table: { category: 'Social' },
    },
    use_timed_date: {
      control: 'boolean',
      description: 'Use semantic time element',
      table: { category: 'SEO' },
    },
  },
};

export default meta;
type Story = StoryObj<ArticleTitleArgs>;

export const Default: Story = {
  args: {
    article_title: 'The Future of Cloud Gaming: A Comprehensive Analysis',
    label: [
      { text: 'Gaming', type: 'category' },
      { text: 'Technology', type: 'tag' },
    ],
    published_date: 'Jan 15, 2024',
    published_at: 'Published',
    updated_date: null,
    updated_at: 'Updated',
    sponso: false,
    sponso_text: 'Sponsored',
    share: 'Share',
    use_timed_date: false,
  },
};

export const WithUpdateDate: Story = {
  args: {
    article_title: 'Breaking: Major Game Studio Announces New Title',
    label: [{ text: 'Breaking News', type: 'breaking' }],
    published_date: 'Jan 10, 2024',
    published_at: 'Published',
    updated_date: 'Jan 15, 2024',
    updated_at: 'Updated',
    sponso: false,
    share: 'Share',
    use_timed_date: false,
  },
};

export const SponsoredContent: Story = {
  args: {
    article_title: 'Best Gaming Laptops for 2024: Ultimate Buyer\'s Guide',
    label: [
      { text: 'Reviews', type: 'category' },
      { text: 'Hardware', type: 'tag' },
    ],
    published_date: 'Jan 12, 2024',
    published_at: 'Published',
    updated_date: null,
    sponso: true,
    sponso_text: 'Sponsored Content',
    share: 'Share',
    use_timed_date: false,
  },
};

export const NoLabels: Story = {
  args: {
    article_title: 'Minimalist Article Without Category Labels',
    label: [],
    published_date: 'Jan 10, 2024',
    published_at: 'Published',
    updated_date: null,
    sponso: false,
    share: 'Share',
    use_timed_date: false,
  },
};

export const MultipleLabels: Story = {
  args: {
    article_title: 'Comprehensive E-Sports Tournament Coverage',
    label: [
      { text: 'E-Sports', type: 'category' },
      { text: 'Tournament', type: 'tag' },
      { text: 'Featured', type: 'featured' },
      { text: 'Live', type: 'live' },
    ],
    published_date: 'Jan 14, 2024',
    published_at: 'Published',
    updated_date: null,
    sponso: false,
    share: 'Share',
    use_timed_date: false,
  },
};

export const LongTitle: Story = {
  args: {
    article_title: 'An In-Depth Exploration of Game Development Methodologies: How Independent Studios Are Revolutionizing the Industry Through Innovative Approaches',
    label: [{ text: 'Feature', type: 'category' }],
    published_date: 'Jan 8, 2024',
    published_at: 'Published',
    updated_date: null,
    sponso: false,
    share: 'Share',
    use_timed_date: false,
  },
};

export const WithSemanticTime: Story = {
  args: {
    article_title: 'SEO-Optimized Article with Semantic Time Element',
    label: [{ text: 'SEO', type: 'category' }],
    published_date: '2024-01-15T10:30:00+00:00',
    published_at: 'Published',
    updated_date: null,
    sponso: false,
    share: 'Share',
    use_timed_date: true,
  },
};

export const NoShareButton: Story = {
  args: {
    article_title: 'Article Without Social Sharing',
    label: [{ text: 'News', type: 'category' }],
    published_date: 'Jan 12, 2024',
    published_at: 'Published',
    updated_date: null,
    sponso: false,
    share: null,
    use_timed_date: false,
  },
};

export const SponsoredWithUpdate: Story = {
  args: {
    article_title: 'Product Review: Latest Gaming Monitor',
    label: [
      { text: 'Review', type: 'category' },
      { text: 'Hardware', type: 'tag' },
    ],
    published_date: 'Dec 20, 2023',
    published_at: 'Published',
    updated_date: 'Jan 15, 2024',
    updated_at: 'Updated',
    sponso: true,
    sponso_text: 'Sponsored',
    share: 'Share',
    use_timed_date: false,
  },
};

export const CustomDateLabels: Story = {
  args: {
    article_title: 'Article with Custom Date Wording',
    label: [{ text: 'Opinion', type: 'category' }],
    published_date: 'Jan 5, 2024',
    published_at: 'Posted on',
    updated_date: 'Jan 10, 2024',
    updated_at: 'Last modified',
    sponso: false,
    share: 'Share',
    use_timed_date: false,
  },
};
