/**
 * Article Headline Module Stories
 *
 * Article page headline/intro container.
 * 
 * This simple module displays an article headline or introduction text,
 * typically positioned at the top of article pages before the main content.
 * Content is expected to contain raw HTML markup.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../../.storybook/types';

interface ArticleHeadlineArgs {
  articlePage_headline: string;
  classes: string[];
}

const meta: Meta<ArticleHeadlineArgs> = {
  title: 'Modules/Article/Article Headline',
  parameters: {
    docs: {
      description: {
        component: `Article headline container for intro text or lead paragraphs.
        
**Features:**
- Simple container for article headline HTML
- Supports rich HTML content
- Default margin/container classes
- Customizable CSS classes

**Usage in projects:**
\`\`\`twig
{% include '@Ui/module/article/article_headline.html.twig' with {
    articlePage_headline: '<p>This is the article introduction...</p>',
    classes: ['mg-content']
} only %}
\`\`\``,
      },
    },
    twig: {
      template: 'module/article/article_headline.html.twig',
      renderMode: 'include',
      componentPath: 'module/article/article_headline',
      argsToContext: (args) => args,
    },
  } as StoryParameters,
  argTypes: {
    articlePage_headline: {
      control: 'text',
      description: 'Headline HTML content',
      table: { category: 'Content' },
    },
    classes: {
      control: 'object',
      description: 'CSS classes for container',
      table: { category: 'Style' },
    },
  },
};

export default meta;
type Story = StoryObj<ArticleHeadlineArgs>;

export const Default: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151;">Discover the latest insights in gaming technology and how it\'s shaping the future of interactive entertainment.</p>',
    classes: ['mg-content'],
  },
};

export const ShortHeadline: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151;">Breaking news from the gaming industry.</p>',
    classes: ['mg-content'],
  },
};

export const LongHeadline: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151;">In this comprehensive analysis, we explore the evolution of open-world game design over the past decade, examining how studios have pushed the boundaries of player freedom, environmental storytelling, and emergent gameplay mechanics to create more immersive and engaging experiences than ever before.</p>',
    classes: ['mg-content'],
  },
};

export const WithFormatting: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151;"><strong>Editor\'s Note:</strong> The following article contains spoilers for <em>The Legend of Zelda: Tears of the Kingdom</em>. Read at your own discretion.</p>',
    classes: ['mg-content'],
  },
};

export const MultiParagraph: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151; margin-bottom: 1rem;">The gaming landscape has changed dramatically in recent years.</p><p style="font-size: 18px; line-height: 1.6; color: #374151;">Here\'s what you need to know about the latest developments.</p>',
    classes: ['mg-content'],
  },
};

export const WithLinks: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151;">Following our <a href="/previous-article" style="color: #2563eb; text-decoration: underline;">previous coverage</a>, we dive deeper into the technical aspects of next-gen gaming.</p>',
    classes: ['mg-content'],
  },
};

export const CustomClasses: Story = {
  args: {
    articlePage_headline: '<p style="font-size: 18px; line-height: 1.6; color: #374151;">This headline has custom styling classes applied.</p>',
    classes: ['mg-content', 'custom-headline', 'featured'],
  },
};

export const NoContent: Story = {
  args: {
    articlePage_headline: '',
    classes: ['mg-content'],
  },
};
