/**
 * Title Component Stories
 *
 * Section or page title with optional subtitle, link, and logo.
 *
 * Uses the actual @Ui/component/title.html.twig template through a wrapper
 * that includes the template with explicit variables. Unlike macro-based
 * components, Title uses flat Twig variables passed directly.
 *
 * Template structure:
 * - title.html.twig expects flat variables (title_text, type_page, etc.)
 * - Story wrapper includes template with { ... } only syntax
 * - type_page switches between section (default) and page title styles
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { TitleMarkup, StoryParameters } from '../../.storybook/types';

interface TitleArgs {
  title_text: string;
  type_page?: boolean;
  title_markup?: TitleMarkup | 'div';
  title_link?: string;
  subtitle_text?: string;
  title_uppercase?: boolean;
  title_text_w_roller?: boolean;
}

// Component metadata
const meta: Meta<TitleArgs> = {
  title: 'Components/Title',
  parameters: {
    docs: {
      description: {
        component:
          'Section or page title with optional subtitle, link, and logo.',
      },
    },
    twig: {
      template: 'component/title.html.twig',
      renderMode: 'include',
      componentPath: 'title',
    },
  } as StoryParameters,
  argTypes: {
    // Core
    title_text: {
      control: 'text',
      description: 'Main title text (required)',
      table: { category: 'Core' },
    },
    type_page: {
      control: 'boolean',
      description: 'Use page title style (larger) vs section title',
      table: { category: 'Core' },
    },
    title_markup: {
      control: 'select',
      options: ['div', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6'],
      description: 'HTML element for title (h1-h6 adds ARIA heading)',
      table: { category: 'Core' },
    },
    // Optional content
    title_link: {
      control: 'text',
      description: 'URL to wrap title in anchor',
      table: { category: 'Content' },
    },
    subtitle_text: {
      control: 'text',
      description: 'Subtitle text below title',
      table: { category: 'Content' },
    },
    // Visual
    title_uppercase: {
      control: 'boolean',
      description: 'Uppercase text transform',
      table: { category: 'Visual' },
    },
    title_text_w_roller: {
      control: 'boolean',
      description: 'Apply roller animation styling to title text',
      table: { category: 'Visual' },
    },
  },
};

export default meta;
type Story = StoryObj<TitleArgs>;

// Playground: All controls available, minimal defaults
export const Playground: Story = {
  args: {
    title_text: 'Title Component',
  },
};

// Section title (default mode)
export const SectionTitle: Story = {
  args: {
    title_text: 'Section Title',
    type_page: false,
    title_markup: 'h2',
  },
};

// Page title (larger styling)
export const PageTitle: Story = {
  args: {
    title_text: 'Page Title',
    type_page: true,
    title_markup: 'h1',
  },
};

// Title with subtitle
export const WithSubtitle: Story = {
  args: {
    title_text: 'Main Title',
    subtitle_text: 'Supporting subtitle text',
  },
};

// Title as a link
export const WithLink: Story = {
  args: {
    title_text: 'Linked Title',
    title_link: '#section',
    title_markup: 'h3',
  },
};

// Title with roller animation styling
export const WithRoller: Story = {
  args: {
    title_text: 'Animated Title',
    title_text_w_roller: true,
    title_markup: 'h2',
  },
};
