/**
 * Thumbnail Component Stories
 *
 * Responsive image component with optional link, play icon, labels, and figcaption.
 *
 * Uses the actual @Ui/component/thumbnail.html.twig template through a wrapper
 * that invokes the thumb macro. The macro accepts an object with image configuration,
 * link options, visual modifiers, and overlay elements.
 *
 * Template structure:
 * - thumbnail.html.twig defines the {% macro thumb(obj) %}
 * - Story wrapper imports and calls the macro with Storybook args
 * - Supports aspect ratios (16:9, 3:4, 1:1), cover mode, play icon
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { LinkTarget, StoryParameters } from '../../.storybook/types';

// Placeholder image for demos
const PLACEHOLDER_IMAGE = 'https://picsum.photos/640/360';

interface ThumbnailArgs {
  picture: string;
  alt: string;
  width?: number;
  height?: number;
  href?: string;
  target?: LinkTarget | '';
  ratio?: '' | '16:9' | '3:4' | '1:1';
  cover?: boolean;
  play?: boolean;
  figcaption?: string;
}

// Component metadata
const meta: Meta<ThumbnailArgs> = {
  title: 'Components/Thumbnail',
  parameters: {
    docs: {
      description: {
        component:
          'Responsive image component with optional link, play icon, labels, and figcaption.',
      },
    },
    twig: {
      template: 'component/thumbnail.html.twig',
      renderMode: 'macro',
      macroName: 'thumb',
      macroParam: 'obj',
      componentPath: 'thumbnail',
      argsToContext: (args) => ({ obj: args }),
    },
  } as StoryParameters,
  argTypes: {
    // Core (image)
    picture: {
      control: 'text',
      description: 'Image source URL',
      table: { category: 'Core' },
    },
    alt: {
      control: 'text',
      description: 'Image alt text',
      table: { category: 'Core' },
    },
    width: {
      control: 'number',
      description: 'Image width in pixels',
      table: { category: 'Core' },
    },
    height: {
      control: 'number',
      description: 'Image height in pixels',
      table: { category: 'Core' },
    },
    // Link
    href: {
      control: 'text',
      description: 'Link URL wrapping the thumbnail',
      table: { category: 'Link' },
    },
    target: {
      control: 'select',
      options: ['', '_self', '_blank'],
      description: 'Link target attribute',
      table: { category: 'Link' },
      if: { arg: 'href', truthy: true },
    },
    // Visual options
    ratio: {
      control: 'select',
      options: ['', '16:9', '3:4', '1:1'],
      description: 'Aspect ratio constraint',
      table: { category: 'Visual' },
    },
    cover: {
      control: 'boolean',
      description: 'Use object-fit: cover (default true)',
      table: { category: 'Visual' },
    },
    play: {
      control: 'boolean',
      description: 'Show play icon overlay (for video thumbnails)',
      table: { category: 'Visual' },
    },
    figcaption: {
      control: 'text',
      description: 'Caption text below image',
      table: { category: 'Visual' },
    },
  },
};

export default meta;
type Story = StoryObj<ThumbnailArgs>;

// Playground: All controls available, minimal defaults
export const Playground: Story = {
  args: {
    picture: PLACEHOLDER_IMAGE,
    alt: 'Placeholder',
    width: 640,
    height: 360,
  },
};

// Basic thumbnail
export const Basic: Story = {
  args: {
    picture: PLACEHOLDER_IMAGE,
    alt: 'Basic image',
    width: 320,
    height: 180,
  },
};

// Thumbnail with link
export const WithLink: Story = {
  args: {
    picture: PLACEHOLDER_IMAGE,
    alt: 'Linked image',
    href: '#article',
    width: 320,
    height: 180,
  },
};

// 16:9 aspect ratio
export const Ratio16x9: Story = {
  args: {
    picture: PLACEHOLDER_IMAGE,
    alt: '16:9 image',
    ratio: '16:9',
    width: 320,
    height: 180,
  },
};

// Video thumbnail with play icon
export const WithPlayIcon: Story = {
  args: {
    picture: PLACEHOLDER_IMAGE,
    alt: 'Video thumbnail',
    play: true,
    href: '#video',
    width: 320,
    height: 180,
  },
};

// Thumbnail with caption
export const WithCaption: Story = {
  args: {
    picture: PLACEHOLDER_IMAGE,
    alt: 'Captioned image',
    figcaption: 'Photo credit: Example',
    width: 320,
    height: 180,
  },
};
