/**
 * Subtitle Component Stories
 *
 * Secondary text element, often used below titles or as section dividers.
 *
 * Uses the actual @Ui/component/subtitle.html.twig template through a wrapper
 * that includes the template with explicit variables. Subtitle supports optional
 * linking and custom HTML markup.
 *
 * Template structure:
 * - subtitle.html.twig expects flat variables (subtitle, href, markup)
 * - Story wrapper includes template with { ... } only syntax
 * - When href is provided, content is wrapped in anchor element
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../.storybook/types';

interface SubtitleArgs {
  subtitle: string;
  href?: string;
  markup?: 'div' | 'span' | 'p';
}

// Component metadata
const meta: Meta<SubtitleArgs> = {
  title: 'Components/Subtitle',
  parameters: {
    docs: {
      description: {
        component:
          'Secondary text element, often used below titles or as section dividers.',
      },
    },
    twig: {
      template: 'component/subtitle.html.twig',
      renderMode: 'include',
      componentPath: 'subtitle',
    },
  } as StoryParameters,
  argTypes: {
    // Core
    subtitle: {
      control: 'text',
      description: 'Subtitle text content (required)',
      table: { category: 'Core' },
    },
    // Link
    href: {
      control: 'text',
      description: 'Optional link URL (wraps content in anchor)',
      table: { category: 'Link' },
    },
    // Visual
    markup: {
      control: 'select',
      options: ['div', 'span', 'p'],
      description: 'HTML element wrapper',
      table: { category: 'Visual' },
    },
  },
};

export default meta;
type Story = StoryObj<SubtitleArgs>;

// Playground: All controls available, minimal defaults
export const Playground: Story = {
  args: {
    subtitle: 'Subtitle text',
  },
};

// Default subtitle without link
export const Default: Story = {
  args: {
    subtitle: 'A descriptive subtitle',
  },
};

// Subtitle with link
export const WithLink: Story = {
  args: {
    subtitle: 'Click for more',
    href: '#more',
  },
};

// Subtitle as span element
export const AsSpan: Story = {
  args: {
    subtitle: 'Inline subtitle',
    markup: 'span',
  },
};
