/**
 * Share Component Stories
 *
 * Share button with native Web Share API integration (requires js-share JS module).
 *
 * Uses the actual @Ui/component/share.html.twig template through a wrapper
 * that includes the template with explicit variables. The component provides
 * data attributes for the js-share module to use with navigator.share().
 *
 * Template structure:
 * - share.html.twig expects flat variables (data_title, data_text, data_href)
 * - Story wrapper includes template with { ... } only syntax
 * - Data attributes are rendered for js-share JavaScript module
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../.storybook/types';

interface ShareArgs {
  data_title?: string;
  data_text?: string;
  data_href?: string;
  txt_wording?: string;
}

// Component metadata
const meta: Meta<ShareArgs> = {
  title: 'Components/Share',
  parameters: {
    viewport: { defaultViewport: 'mobile1' },
    docs: {
      description: {
        component:
          'Share button with native Web Share API integration (requires js-share JS module).',
      },
    },
    twig: {
      template: 'component/share.html.twig',
      renderMode: 'include',
      componentPath: 'share',
    },
  } as StoryParameters,
  argTypes: {
    // Data attributes
    data_title: {
      control: 'text',
      description: 'Share title (data-title attribute)',
      table: { category: 'Data' },
    },
    data_text: {
      control: 'text',
      description: 'Share description (data-text attribute)',
      table: { category: 'Data' },
    },
    data_href: {
      control: 'text',
      description: 'Share URL (data-href attribute)',
      table: { category: 'Data' },
    },
    // Visual
    txt_wording: {
      control: 'text',
      description: 'Button label text (optional)',
      table: { category: 'Visual' },
    },
  },
};

export default meta;
type Story = StoryObj<ShareArgs>;

// Playground: All controls available, minimal defaults
export const Playground: Story = {
  args: {
    data_title: 'Article Title',
    data_href: 'https://example.com/article',
  },
};

// Default share button with all data attributes
export const Default: Story = {
  args: {
    data_title: 'Share This Article',
    data_text: 'Check out this interesting article',
    data_href: 'https://example.com',
  },
};

// Share button with visible text label
export const WithText: Story = {
  args: {
    data_title: 'Share',
    data_href: 'https://example.com',
    txt_wording: 'Share this page',
  },
};

// Share button with icon only (no text)
export const IconOnly: Story = {
  args: {
    data_title: 'Quick Share',
    data_href: 'https://example.com',
  },
};
