/**
 * Label Component Stories
 *
 * Inline label/tag component for categorization, status indicators, and badges.
 *
 * Uses the actual @Ui/component/label.html.twig template through a wrapper
 * that invokes the label macro. The macro accepts an object with configuration
 * options for text and content wrapper.
 *
 * Template structure:
 * - label.html.twig defines the {% macro label(obj) %}
 * - Story wrapper imports and calls the macro with Storybook args
 *
 * NOTE: The Twig template supports additional options (type, color, background, size)
 * that generate CSS classes, but the corresponding SCSS mixins are not invoked in
 * ui-bundle. Consuming projects must invoke those mixins in their own SCSS to enable
 * these features. Only controls with working CSS in ui-bundle are exposed here.
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { StoryParameters } from '../../.storybook/types';

interface LabelArgs {
  value: string;
  has_content_holder?: boolean;
}

// Component metadata
const meta: Meta<LabelArgs> = {
  title: 'Components/Label',
  parameters: {
    docs: {
      description: {
        component:
          'Inline label/tag component for categorization, status indicators, and badges.',
      },
    },
    twig: {
      template: 'component/label.html.twig',
      renderMode: 'macro',
      macroName: 'label',
      macroParam: 'obj',
      componentPath: 'label',
      argsToContext: (args) => ({ obj: args }),
    },
  } as StoryParameters,
  argTypes: {
    // Core
    value: {
      control: 'text',
      description: 'Label text content (use instead of deprecated "text")',
      table: { category: 'Core' },
    },
    // Visual
    has_content_holder: {
      control: 'boolean',
      description: 'Wrap content in span.label-content',
      table: { category: 'Visual' },
    },
  },
};

export default meta;
type Story = StoryObj<LabelArgs>;

// Playground: All controls available, minimal defaults
export const Playground: Story = {
  args: {
    value: 'Label',
  },
};

// Default label with just text
export const Default: Story = {
  args: {
    value: 'Category',
  },
};

// Label with content holder wrapper
export const WithContentHolder: Story = {
  args: {
    value: 'Wrapped Content',
    has_content_holder: true,
  },
};
