/**
 * Button Component Stories
 *
 * Real UI-Bundle button component rendered via php-wasm.
 *
 * Uses the actual @Ui/component/button.html.twig template through a wrapper
 * that invokes the button macro. Symfony Twig Bridge extensions are registered
 * with mock services for Storybook rendering.
 *
 * Template structure:
 * - button.html.twig defines the {% macro button(button_obj) %}
 * - Story wrapper imports and calls the macro with Storybook args
 * - @Ui namespace resolves to real templates directory
 */

import type { Meta, StoryObj } from '@storybook/react';
import type { ButtonSize, LinkTarget, StoryParameters } from '../../.storybook/types';

interface ButtonArgs {
  text: string;
  size?: ButtonSize;
  href?: string;
  target?: LinkTarget;
  icon?: string;
  extraClass?: string;
  disabled?: boolean;
}

// Component metadata
const meta: Meta<ButtonArgs> = {
  title: 'Components/Button',
  parameters: {
    docs: {
      description: {
        component: 'Versatile button component supporting multiple styles, sizes, and states.',
      },
    },
    twig: {
      template: 'component/button.html.twig',
      renderMode: 'macro',
      macroName: 'button',
      macroParam: 'button_obj',
      componentPath: 'button',
      argsToContext: (args) => ({ button_obj: args }),
    },
  } as StoryParameters,
  argTypes: {
    // Core props
    text: {
      control: 'text',
      description: 'Button text label',
      table: { category: 'Core' },
    },
    size: {
      control: 'select',
      options: ['sm', 'md', 'lg'],
      description: 'Button size',
      table: { category: 'Core' },
    },
    // Link props
    href: {
      control: 'text',
      description: 'Link URL (makes button an anchor)',
      table: { category: 'Link' },
    },
    target: {
      control: 'select',
      options: ['_self', '_blank'],
      description: 'Link target attribute',
      table: { category: 'Link' },
      if: { arg: 'href', truthy: true },
    },
    // Visual props
    icon: {
      control: 'text',
      description: 'Icon name (adds icon before text)',
      table: { category: 'Visual' },
    },
    extraClass: {
      control: 'text',
      description: 'Additional CSS classes',
      table: { category: 'Visual' },
    },
    disabled: {
      control: 'boolean',
      description: 'Disable button interaction',
      table: { category: 'Visual' },
    },
  },
};

export default meta;
type Story = StoryObj<ButtonArgs>;

// Playground variant for experimentation
export const Playground: Story = {
  args: {
    text: 'Button',
  },
};

export const Primary: Story = {
  args: {
    text: 'Primary Button',
  },
};

export const Secondary: Story = {
  args: {
    text: 'Secondary Button',
    extraClass: 'btn-secondary',
  },
};

export const WithIcon: Story = {
  args: {
    text: 'Download',
    icon: 'download',
  },
};

export const AsLink: Story = {
  args: {
    text: 'Visit Site',
    href: 'https://example.com',
    target: '_blank',
  },
};

export const Disabled: Story = {
  args: {
    text: 'Disabled Button',
    disabled: true,
  },
};
